/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.tooling.client.api.component.location;

import org.mule.tooling.client.api.component.ComponentIdentifier;
import org.mule.tooling.client.api.component.TypedComponentIdentifier;

import java.util.List;
import java.util.Optional;

import static java.util.Optional.ofNullable;

/**
 * Provides information about the location of a component within an application.
 *
 * A component location describes where the component is defined in the configuration of the artifact.
 *
 * For instance:
 * <ul>
 * <li>COMPONENT_NAME - global component defined with name COMPONENT_NAME</li>
 * <li>FLOW_NAME/source - a source defined within a flow</li>
 * <li>FLOW_NAME/processors/0 - the first processor defined within a flow with name FLOW_NAME</li>
 * <li>FLOW_NAME/processors/4/1 - the first processors defined inside another processor which is positioned fifth within a flow
 * with name FLOW_NAME</li>
 * <li>FLOW_NAME/errorHandler/0 - the first on-error within the error handler</li>
 * <li>FLOW_NAME/0/errorHandler/3 - the third on-error within the error handler of the first element of the flow with name
 * FLOW_NAME</li>
 * </ul>
 *
 * @since 1.0
 */
public class ComponentLocation {

  private String location;
  private String fileName;
  private Integer lineInFile;
  private TypedComponentIdentifier typedComponentIdentifier;
  private List<LocationPart> locationParts;

  private ComponentLocation() {}

  public ComponentLocation(String location, String fileName, Integer lineInFile,
                           TypedComponentIdentifier typedComponentIdentifier, List<LocationPart> locationParts) {
    this.location = location;
    this.fileName = fileName;
    this.lineInFile = lineInFile;
    this.typedComponentIdentifier = typedComponentIdentifier;
    this.locationParts = locationParts;
  }

  /**
   * @return the unique absolute path of the component in the application.
   */
  public String getLocation() {
    return location;
  }

  /**
   * @return the config file of the application where this component is defined, if it was defined in a config file.
   */
  public Optional<String> getFileName() {
    return ofNullable(fileName);
  }

  /**
   * @return the line number in the config file of the application where this component is defined, if it was defined in a config
   *         file.
   */
  public Optional<Integer> getLineInFile() {
    return ofNullable(lineInFile);
  }

  /**
   * @return the list of parts for the location. The location starts with the global element containing the component and
   *         continues with the next elements within the global element until the last part which is the component specific part.
   */
  public List<LocationPart> getParts() {
    return locationParts;
  }

  /**
   * @return the {@link ComponentIdentifier} of the component associated with this location
   */
  public TypedComponentIdentifier getComponentIdentifier() {
    return typedComponentIdentifier;
  }

}
