/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */

package org.mule.tooling.client.api.configuration.ssl;

import static java.util.Objects.requireNonNull;
import org.mule.tooling.client.internal.configuration.ssl.ImmutableTrustStoreConfig;

import java.io.File;

/**
 * Configuration for a trust store.
 *
 * @sicne 1.0
 */
public interface TrustStoreConfig {

  /**
   * @return trust store {@link File}.
   */
  File getTrustStoreFile();

  /**
   * @return the password for the trust store.
   */
  String getTrustStorePassword();

  /**
   * @return a {@link Builder} to build a {@link TrustStoreConfig}.
   */
  static Builder builder() {
    return new Builder();
  }

  class Builder {

    private File trustStoreFile;
    private String trustStorePassword;

    /**
     * Sets the trust store {@link File}.
     *
     * @param trustStoreFile key store {@link File}.
     * @return this
     */
    public Builder withTrustStoreFile(File trustStoreFile) {
      requireNonNull(trustStoreFile, "trustStoreFile cannot be null");
      this.trustStoreFile = trustStoreFile;
      return this;
    }

    /**
     * Sets the password for the trust store.
     *
     * @param trustStorePassword password for the trust store.
     * @return this
     */
    public Builder withTrustStorePassword(String trustStorePassword) {
      requireNonNull(trustStorePassword, "trustStorePassword cannot be null");
      this.trustStorePassword = trustStorePassword;
      return this;
    }

    /**
     * Builds the {@link TrustStoreConfig} object.
     *
     * @return {@link TrustStoreConfig} with the value sets.
     */
    public TrustStoreConfig build() {
      return new ImmutableTrustStoreConfig(this.trustStoreFile, this.trustStorePassword);
    }

  }

}
