/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */

package org.mule.tooling.client.internal.serialization;

import static org.mule.tooling.client.internal.serialization.TypeIdAnnotationMapper.TYPE_ANNOTATIONS_MAPPING;
import org.mule.metadata.api.annotation.TypeAnnotation;
import org.mule.tooling.client.internal.persistence.TypeAnnotationEntryKey;

import com.thoughtworks.xstream.XStream;
import com.thoughtworks.xstream.converters.MarshallingContext;
import com.thoughtworks.xstream.converters.UnmarshallingContext;
import com.thoughtworks.xstream.converters.collections.MapConverter;
import com.thoughtworks.xstream.io.HierarchicalStreamReader;
import com.thoughtworks.xstream.io.HierarchicalStreamWriter;
import com.thoughtworks.xstream.mapper.Mapper;

import java.util.LinkedHashMap;
import java.util.Map;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Serialization support that allows to serialize/deserialze objects from this API.
 *
 * @since 1.0
 */
public final class SerializationUtils {

  private static Logger LOGGER = LoggerFactory.getLogger(SerializationUtils.class);

  private static XStream createXStream() {
    final XStream xStream = XStreamFactory.createXStream();
    xStream.registerConverter(new TypeAnnotationsMapConverter(xStream.getMapper()));
    xStream.registerConverter(new XmlArtifactDeserializationRequestConverter(xStream.getMapper()));
    xStream.ignoreUnknownElements();
    return xStream;
  }

  private static class TypeAnnotationsMapConverter extends MapConverter {

    public TypeAnnotationsMapConverter(Mapper mapper) {
      super(mapper);
    }

    @Override
    public void marshal(Object source, HierarchicalStreamWriter writer, MarshallingContext context) {
      Map map = (Map) source;
      if (map.isEmpty()) {
        super.marshal(source, writer, context);
        return;
      }
      final Map.Entry entry = (Map.Entry) map.entrySet().iterator().next();
      if (entry.getKey() instanceof Class) {
        if (TypeAnnotation.class.isAssignableFrom((Class<?>) entry.getKey())) {
          Map<Class, TypeAnnotation> typeAnnotationMap = (Map<Class, TypeAnnotation>) map;

          Map<TypeAnnotationEntryKey, TypeAnnotation> toTypeAnnotationTransformedMap = new LinkedHashMap<>();
          for (Map.Entry<Class, TypeAnnotation> e : typeAnnotationMap.entrySet()) {
            if (e.getValue().isPublic()) {
              toTypeAnnotationTransformedMap.put(new TypeAnnotationEntryKey(e.getValue().getName()), e.getValue());
            }
          }
          super.marshal(toTypeAnnotationTransformedMap, writer, context);
        }
      } else {
        super.marshal(source, writer, context);
      }
    }

    @Override
    protected void putCurrentEntryIntoMap(HierarchicalStreamReader reader, UnmarshallingContext context,
                                          Map map, Map target) {
      reader.moveDown();
      Object key = readItem(reader, context, map);
      reader.moveUp();

      Object value = null;
      reader.moveDown();
      if (key instanceof TypeAnnotationEntryKey) {
        String typeId = ((TypeAnnotationEntryKey) key).getTypeId();
        if (TYPE_ANNOTATIONS_MAPPING.containsKey(typeId)) {
          key = TYPE_ANNOTATIONS_MAPPING.get(typeId);
        } else {
          LOGGER.warn("Unknown typeId:'{}' when unmarshalling TypeAnnotation", typeId);
          key = null;
        }
      }
      if (key != null) {
        value = readItem(reader, context, map);
      }
      reader.moveUp();

      if (key != null && value != null) {
        target.put(key, value);
      }
    }

  }

  public static String serialize(Object object) {
    return createXStream().toXML(object);
  }

  public static <T> T deserialize(String xml) {
    return (T) createXStream().fromXML(xml);
  }

}
