/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */

package org.mule.tooling.client.internal.serialization;

import static org.mule.tooling.client.internal.serialization.TypeIdAnnotationMapper.TYPE_ANNOTATIONS_MAPPING;
import org.mule.metadata.api.annotation.TypeAnnotation;
import org.mule.metadata.api.model.impl.DefaultArrayType;
import org.mule.tooling.client.api.exception.ToolingException;
import org.mule.tooling.client.internal.persistence.TypeAnnotationEntryKey;

import com.esotericsoftware.kryo.Kryo;
import com.esotericsoftware.kryo.io.Input;
import com.esotericsoftware.kryo.io.Output;
import com.esotericsoftware.kryo.serializers.MapSerializer;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.util.Base64;
import java.util.LinkedHashMap;
import java.util.Map;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Serialization support that allows to serialize/deserialze objects from this API.
 *
 * @since 1.0
 */
public final class KryoClientSerializer implements Serializer {

  public static String NAME = "kryo";

  private static Logger LOGGER = LoggerFactory.getLogger(KryoClientSerializer.class);

  private static Kryo createKryo() {
    final Kryo kryo = KryoFactory.createKryo();
    kryo.addDefaultSerializer(DefaultArrayType.class, new DefaultArrayTypeSerializer());
    // org.mule.metadata.api.builder.AbstractBuilder.annotations uses a LinkedHashMap
    kryo.register(LinkedHashMap.class, new TypeAnnotationsMapSerializer());
    return kryo;
  }

  @Override
  public String serialize(Object object) {
    try (ByteArrayOutputStream byteArrayOutputStream = new ByteArrayOutputStream()) {
      try (Output output = new Output(byteArrayOutputStream)) {
        createKryo().writeClassAndObject(output, object);
      }
      return Base64.getEncoder().encodeToString(byteArrayOutputStream.toByteArray());
    } catch (IOException e) {
      throw new ToolingException("Error while creating object from serialization", e);
    }
  }

  @Override
  public <T> T deserialize(String content) {
    try (Input input = new Input(new ByteArrayInputStream(Base64.getDecoder().decode(content)))) {
      return (T) createKryo().readClassAndObject(input);
    }
  }

  private static class TypeAnnotationsMapSerializer extends MapSerializer {

    @Override
    public void write(Kryo kryo, Output output, Map map) {
      if (map.isEmpty()) {
        super.write(kryo, output, map);
        return;
      }
      final Map.Entry entry = (Map.Entry) map.entrySet().iterator().next();
      if (entry.getKey() instanceof Class && TypeAnnotation.class.isAssignableFrom((Class<?>) entry.getKey())) {
        Map<Class, TypeAnnotation> typeAnnotationMap = (Map<Class, TypeAnnotation>) map;

        Map<TypeAnnotationEntryKey, TypeAnnotation> toTypeAnnotationTransformedMap = new LinkedHashMap<>();
        for (Map.Entry<Class, TypeAnnotation> e : typeAnnotationMap.entrySet()) {
          if (e.getValue().isPublic()) {
            toTypeAnnotationTransformedMap.put(new TypeAnnotationEntryKey(e.getValue().getName()), e.getValue());
          }
        }
        map = toTypeAnnotationTransformedMap;
      }
      super.write(kryo, output, map);
    }

    @Override
    public Map read(Kryo kryo, Input input, Class<Map> type) {
      Map map = create(kryo, input, type);
      int length = input.readInt(true);

      kryo.reference(map);

      for (int i = 0; i < length; i++) {
        Object key = kryo.readClassAndObject(input);

        if (key instanceof TypeAnnotationEntryKey) {
          String typeId = ((TypeAnnotationEntryKey) key).getTypeId();
          if (TYPE_ANNOTATIONS_MAPPING.containsKey(typeId)) {
            key = TYPE_ANNOTATIONS_MAPPING.get(typeId);
          } else {
            LOGGER.warn("Unknown typeId:'{}' when unmarshalling TypeAnnotation", typeId);
            key = null;
          }
        }
        Object value = null;
        if (key != null) {
          value = kryo.readClassAndObject(input);
        }

        if (key != null && value != null) {
          map.put(key, value);
        }
      }
      return map;
    }
  }

}
