/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */

package org.mule.tooling.client.internal.serialization;

import static java.util.Collections.synchronizedList;
import static org.mule.tooling.client.internal.serialization.KryoFactory.externalizableKryo;

import org.mule.tooling.client.api.exception.ToolingException;
import org.mule.tooling.client.internal.serialization.mapping.TypeAnnotationMapWriter;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Base64;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;

import com.esotericsoftware.kryo.Kryo;
import com.esotericsoftware.kryo.io.Input;
import com.esotericsoftware.kryo.io.Output;
import com.esotericsoftware.kryo.serializers.MapSerializer;

/**
 * Serialization support that allows to serialize/deserialze objects from this API.
 *
 * @since 1.0
 */
public final class KryoClientSerializer implements Serializer {

  public static String NAME = "kryo";

  private ClassLoader sourceClassLoader;
  private ClassLoader targetClassLoader;
  private List<Class> typeAnnotationsNotSupportedByToolingRuntimeClient;

  public KryoClientSerializer(ClassLoader sourceClassLoader, ClassLoader targetClassLoader) {
    this.sourceClassLoader = sourceClassLoader;
    this.targetClassLoader = targetClassLoader;
    this.typeAnnotationsNotSupportedByToolingRuntimeClient = synchronizedList(new ArrayList<>());
  }

  private static Kryo createKryo(ClassLoader sourceClassLoader, ClassLoader targetClassLoader,
                                 List<Class> typeAnnotationsSupportedByImplementation) {
    final Kryo kryo = externalizableKryo(targetClassLoader);
    kryo.setClassLoader(sourceClassLoader);
    // org.mule.metadata.api.builder.AbstractBuilder.annotations uses a LinkedHashMap
    kryo.register(LinkedHashMap.class,
                  new TypeAnnotationsMapSerializer(targetClassLoader, typeAnnotationsSupportedByImplementation));
    return kryo;
  }

  @Override
  public String serialize(Object object) {
    return doSerialize((kryo, objectToBeSerialized, output) -> kryo.writeClassAndObject(output, objectToBeSerialized), object);
  }

  @Override
  public <T> T deserialize(String content) {
    try (Input input = new Input(new ByteArrayInputStream(Base64.getDecoder().decode(content)))) {
      return (T) createKryo(sourceClassLoader, targetClassLoader, typeAnnotationsNotSupportedByToolingRuntimeClient)
          .readClassAndObject(input);
    }
  }

  public String safeSerialize(Object object) {
    return doSerialize((kryo, objectToBeSerialized, output) -> kryo.writeObject(output, objectToBeSerialized), object);
  }

  private String doSerialize(KryoWriteTask writeTask, Object object) {
    try (ByteArrayOutputStream byteArrayOutputStream = new ByteArrayOutputStream()) {
      try (Output output = new Output(byteArrayOutputStream)) {
        writeTask.write(createKryo(sourceClassLoader, targetClassLoader, typeAnnotationsNotSupportedByToolingRuntimeClient),
                        object, output);
      }
      return Base64.getEncoder().encodeToString(byteArrayOutputStream.toByteArray());
    } catch (IOException e) {
      throw new ToolingException("Error while creating object from serialization", e);
    }
  }

  private interface KryoWriteTask {

    void write(Kryo kryo, Object object, Output output);
  }

  private static class TypeAnnotationsMapSerializer extends MapSerializer {

    private TypeAnnotationMapWriter typeAnnotationMapWriter;

    public TypeAnnotationsMapSerializer(ClassLoader targetClassLoader,
                                        List<Class> typeAnnotationsNotSupportedByToolingRuntimeClient) {
      this.typeAnnotationMapWriter =
          new TypeAnnotationMapWriter(targetClassLoader, typeAnnotationsNotSupportedByToolingRuntimeClient);
    }

    @Override
    public void write(Kryo kryo, Output output, Map map) {
      super.write(kryo, output, typeAnnotationMapWriter.write(map));
    }

  }

}
