/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */

package org.mule.tooling.client.internal.serialization;

import static java.util.Arrays.asList;
import static java.util.Collections.EMPTY_LIST;
import static java.util.Collections.EMPTY_MAP;
import static java.util.Collections.EMPTY_SET;
import static java.util.Collections.singleton;
import static java.util.Collections.singletonList;
import static java.util.Collections.singletonMap;

import org.mule.metadata.api.model.impl.DefaultArrayType;

import java.io.File;
import java.io.InputStream;
import java.lang.reflect.InvocationHandler;
import java.util.GregorianCalendar;

import com.esotericsoftware.kryo.Kryo;
import com.esotericsoftware.kryo.Serializer;
import com.esotericsoftware.kryo.io.Input;
import com.esotericsoftware.kryo.io.Output;
import com.esotericsoftware.kryo.serializers.CompatibleFieldSerializer;
import com.esotericsoftware.kryo.serializers.FieldSerializer;
import de.javakaffee.kryoserializers.ArraysAsListSerializer;
import de.javakaffee.kryoserializers.CollectionsEmptyListSerializer;
import de.javakaffee.kryoserializers.CollectionsEmptyMapSerializer;
import de.javakaffee.kryoserializers.CollectionsEmptySetSerializer;
import de.javakaffee.kryoserializers.CollectionsSingletonListSerializer;
import de.javakaffee.kryoserializers.CollectionsSingletonMapSerializer;
import de.javakaffee.kryoserializers.CollectionsSingletonSetSerializer;
import de.javakaffee.kryoserializers.GregorianCalendarSerializer;
import de.javakaffee.kryoserializers.JdkProxySerializer;
import de.javakaffee.kryoserializers.SynchronizedCollectionsSerializer;
import de.javakaffee.kryoserializers.UnmodifiableCollectionsSerializer;
import de.javakaffee.kryoserializers.cglib.CGLibProxySerializer;
import de.javakaffee.kryoserializers.guava.ArrayListMultimapSerializer;
import de.javakaffee.kryoserializers.guava.HashMultimapSerializer;
import de.javakaffee.kryoserializers.guava.ImmutableListSerializer;
import de.javakaffee.kryoserializers.guava.ImmutableMapSerializer;
import de.javakaffee.kryoserializers.guava.ImmutableMultimapSerializer;
import de.javakaffee.kryoserializers.guava.ImmutableSetSerializer;
import de.javakaffee.kryoserializers.guava.LinkedHashMultimapSerializer;
import de.javakaffee.kryoserializers.guava.LinkedListMultimapSerializer;
import de.javakaffee.kryoserializers.guava.ReverseListSerializer;
import de.javakaffee.kryoserializers.guava.TreeMultimapSerializer;
import de.javakaffee.kryoserializers.guava.UnmodifiableNavigableSetSerializer;
import org.objenesis.strategy.StdInstantiatorStrategy;

/**
 * Factory for {@link Kryo}.
 *
 * @since 1.0
 */
public final class KryoFactory {

  private KryoFactory() {}

  /**
   * @return a default {@link Kryo} instance, with all custom serializers configured.
   */
  public static Kryo defaultKryo() {
    Kryo kryo = new Kryo();
    kryo.setDefaultSerializer(CompatibleFieldSerializer.class);
    configure(kryo, null);
    return kryo;
  }

  /**
   * @return an instance of {@link Kryo} configured with the default settings (converters and serializer).
   */
  public static Kryo externalizableKryo() {
    Kryo kryo = new ExternalizableKryo(new CompatibleClassResolver());
    kryo.setDefaultSerializer(CompatibleFieldSerializer.class);
    configure(kryo, null);
    return kryo;
  }

  /**
   * @param targetClassLoader when classes need to be serialized between different versions the target classloader defines which types should be ignored
   * @return an instance of {@link Kryo} configured with the default settings (converters and serializer).
   */
  public static Kryo externalizableKryo(ClassLoader targetClassLoader) {
    ClassLoaderClassSerializerFilter classLoaderClassFieldSerializerFilter =
        new ClassLoaderClassSerializerFilter(targetClassLoader);
    ClassLoaderFieldSerializerFilter classLoaderFieldSerializerFilter = new ClassLoaderFieldSerializerFilter(targetClassLoader);
    Kryo kryo = new ExternalizableKryo(classLoaderClassFieldSerializerFilter, classLoaderFieldSerializerFilter,
                                       new CompatibleClassResolver());
    kryo.setDefaultSerializer(ExtendedCompatibleFieldSerializer.class);
    configure(kryo, targetClassLoader);
    return kryo;
  }

  private static void configure(Kryo kryo, ClassLoader targetClassLoader) {
    kryo.addDefaultSerializer(DefaultArrayType.class, new DefaultArrayTypeSerializer());

    kryo.getFieldSerializerConfig().setCachedFieldNameStrategy(FieldSerializer.CachedFieldNameStrategy.EXTENDED);
    kryo.setInstantiatorStrategy(new DefaultInstantiatorStrategyWithCache(new StdInstantiatorStrategy()));

    kryo.register(asList("").getClass(), new ArraysAsListSerializer());
    kryo.register(EMPTY_LIST.getClass(), new CollectionsEmptyListSerializer());
    kryo.register(EMPTY_MAP.getClass(), new CollectionsEmptyMapSerializer());
    kryo.register(EMPTY_SET.getClass(), new CollectionsEmptySetSerializer());
    kryo.register(singletonList("").getClass(), new CollectionsSingletonListSerializer());
    kryo.register(singleton("").getClass(), new CollectionsSingletonSetSerializer());
    kryo.register(singletonMap("", "").getClass(), new CollectionsSingletonMapSerializer());
    kryo.register(GregorianCalendar.class, new GregorianCalendarSerializer());
    kryo.register(InvocationHandler.class, new JdkProxySerializer());
    UnmodifiableCollectionsSerializer.registerSerializers(kryo);
    SynchronizedCollectionsSerializer.registerSerializers(kryo);

    kryo.register(CGLibProxySerializer.CGLibProxyMarker.class, new CGLibProxySerializer());

    // guava ImmutableList, ImmutableSet, ImmutableMap, ImmutableMultimap, ReverseList, UnmodifiableNavigableSet
    ImmutableListSerializer.registerSerializers(kryo);
    ImmutableSetSerializer.registerSerializers(kryo);
    ImmutableMapSerializer.registerSerializers(kryo);
    ImmutableMultimapSerializer.registerSerializers(kryo);
    ReverseListSerializer.registerSerializers(kryo);
    UnmodifiableNavigableSetSerializer.registerSerializers(kryo);
    // guava ArrayListMultimap, HashMultimap, LinkedHashMultimap, LinkedListMultimap, TreeMultimap
    ArrayListMultimapSerializer.registerSerializers(kryo);
    HashMultimapSerializer.registerSerializers(kryo);
    LinkedHashMultimapSerializer.registerSerializers(kryo);
    LinkedListMultimapSerializer.registerSerializers(kryo);
    TreeMultimapSerializer.registerSerializers(kryo);

    if (isClassAvailable(targetClassLoader, InputStreamKryoSerializer.class.getName())) {
      kryo.register(InputStream.class, new InputStreamKryoSerializer());
    }

    //https://github.com/EsotericSoftware/kryo/issues/136
    kryo.register(File.class, new Serializer() {

      @Override
      public void write(Kryo kryo, Output output, Object object) {
        output.writeString(((File) object).getAbsolutePath());
      }

      @Override
      public Object read(Kryo kryo, Input input, Class type) {
        return new File(input.readString());
      }

    });
  }

  private static boolean isClassAvailable(ClassLoader classLoader, String name) {
    if (classLoader == null) {
      return false;
    }
    try {
      Class clazz = Class.forName(name, false, classLoader);
      return null != clazz;
    } catch (Throwable e) {
      return false;
    }
  }
}
