/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */

package org.mule.tooling.client.internal.serialization;

import static org.apache.commons.io.IOUtils.toByteArray;
import static org.codehaus.plexus.util.ReflectionUtils.setVariableValueInObject;
import org.mule.tooling.client.api.artifact.declaration.request.XmlArtifactDeserializationRequest;
import org.mule.tooling.client.api.descriptors.ArtifactDescriptor;

import com.thoughtworks.xstream.converters.Converter;
import com.thoughtworks.xstream.converters.MarshallingContext;
import com.thoughtworks.xstream.converters.UnmarshallingContext;
import com.thoughtworks.xstream.io.ExtendedHierarchicalStreamWriterHelper;
import com.thoughtworks.xstream.io.HierarchicalStreamReader;
import com.thoughtworks.xstream.io.HierarchicalStreamWriter;
import com.thoughtworks.xstream.mapper.Mapper;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.Base64;
import java.util.List;

/**
 * {@link Converter} for {@link XmlArtifactDeserializationRequest}.
 *
 * @since 1.0
 */
public class XmlArtifactDeserializationRequestConverter implements Converter {

  private Mapper mapper;

  public XmlArtifactDeserializationRequestConverter(Mapper mapper) {
    this.mapper = mapper;
  }

  @Override
  public void marshal(Object source, HierarchicalStreamWriter writer, MarshallingContext context) {
    XmlArtifactDeserializationRequest request = (XmlArtifactDeserializationRequest) source;
    writer.startNode("artifact-source");
    try {
      writer.setValue(Base64.getEncoder().encodeToString(toByteArray(request.getArtifactSource())));
    } catch (IOException e) {
      throw new RuntimeException("Error while serializing inputStream", e);
    }
    writer.endNode();

    writer.startNode("request-timeout");
    writer.setValue(String.valueOf(request.getRequestTimeout()));
    writer.endNode();

    String typeName = mapper.serializedClass(request.getPluginArtifactDescriptors().getClass());
    ExtendedHierarchicalStreamWriterHelper.startNode(writer, typeName, request.getPluginArtifactDescriptors().getClass());
    context.convertAnother(request.getPluginArtifactDescriptors());
    writer.endNode();
  }

  @Override
  public Object unmarshal(HierarchicalStreamReader reader, UnmarshallingContext context) {
    XmlArtifactDeserializationRequest request = new XmlArtifactDeserializationRequest(null, null);
    InputStream artifactSource = null;
    List<ArtifactDescriptor> pluginArtifactDescriptors = null;
    long requestTimeout = -1;

    while (reader.hasMoreChildren()) {
      reader.moveDown();
      if ("artifact-source".equals(reader.getNodeName())) {
        artifactSource = new ByteArrayInputStream(Base64.getDecoder().decode(reader.getValue()));
      } else if ("request-timeout".equals(reader.getNodeName())) {
        requestTimeout = Long.valueOf(reader.getValue());
      } else {
        pluginArtifactDescriptors = (List<ArtifactDescriptor>) context.convertAnother(request, List.class);
      }
      reader.moveUp();
    }

    try {
      // Have to use reflection as the instance has to be created first so xStream can handle cyclic references...
      setVariableValueInObject(request, "artifactSource", artifactSource);
      setVariableValueInObject(request, "pluginArtifactDescriptors", pluginArtifactDescriptors);
      setVariableValueInObject(request, "requestTimeout", requestTimeout);
    } catch (IllegalAccessException e) {
      throw new RuntimeException("Error while setting attributes for xmlArtifactDeserializationRequest using reflection", e);
    }
    return request;
  }

  @Override
  public boolean canConvert(Class type) {
    return XmlArtifactDeserializationRequest.class.isAssignableFrom(type);
  }

}
