/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */

package org.mule.tooling.client.api.extension;

import org.mule.tooling.client.api.descriptors.ArtifactDescriptor;
import org.mule.tooling.client.api.extension.model.ExtensionModel;
import java.io.File;
import java.util.List;
import java.util.Optional;

/**
 * Service for loading {@link ExtensionModel ExtensionModels}.
 *
 * A {@code ExtensionModelService}
 *
 * @since 1.0
 */
public interface ExtensionModelService {

  /**
   * Loads the {@link ExtensionModel} for the given plugin.
   *
   * @param pluginDescriptor {@link ArtifactDescriptor} referencing to the plugin artifact.
   * @return {@link ExtensionModel} for the plugin or {@link Optional#empty()} if the plugin doesn't have an
   *         {@link ExtensionModel}.
   */
  Optional<ExtensionModel> loadExtensionModel(ArtifactDescriptor pluginDescriptor);

  /**
   * Loads the {@link ExtensionModel} for the given plugin.
   *
   * @param plugin {@link File} referencing to the plugin jar file or the unzipped folder.
   * @return {@link ExtensionModel} for the plugin or {@link Optional#empty()} if the plugin doesn't have an
   *         {@link ExtensionModel}.
   */
  Optional<ExtensionModel> loadExtensionModel(File plugin);

  /**
   * Loads the schema for the given plugin.
   *
   * @param pluginDescriptor {@link ArtifactDescriptor} referencing to the plugin artifact.
   * @return {@link String} with the schema content for the plugin or {@link Optional#empty()} if the plugin doesn't have an
   *         {@link ExtensionModel}.
   */
  Optional<String> loadExtensionSchema(ArtifactDescriptor pluginDescriptor);

  /**
   * Loads the schema for the given plugin.
   *
   * @param plugin {@link File} referencing to the plugin jar file or the unzipped folder.
   * @return an {@link String} with the schema content or {@link Optional#empty()} if the plugin doesn't have an
   * {@link ExtensionModel}.
   */
  Optional<String> loadExtensionSchema(File plugin);

  /**
   * Loads the {@link ExtensionModel}s for the Mule Runtime core components.
   *
   * @return a {@link List} with the {@link ExtensionModel}s for Mule Runtime core components.
   */
  List<ExtensionModel> loadMuleExtensionModels();

}
