/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */

package org.mule.tooling.client.api.artifact.resources;

import java.io.InputStream;
import java.net.URL;
import java.util.Optional;

/**
 * A resource loader that allows to load {@link org.mule.tooling.client.api.artifact.ToolingArtifact} resources
 * with the {@code resource::} prefix or just as relative paths.
 * This resource loader honors the class loader filtering rules defined by the application and plugins as in Runtime
 * environments.
 *
 * @since 1.2.0, 1.1.4
 */
public interface ResourceLoader {

  /**
   * Returns an {@link InputStream} with the resource's data ready to be consumed.
   *
   * @param resource the resource to be found.
   * @return a stream to read data from the resource.
   */
  InputStream getResourceAsStream(String resource);

  /**
   * Returns the resources {@link URL}, useful when the actual resource will be loaded by another component.
   *
   * @param resource the resource to be found.
   * @return A URL pointing to the resource.
   */
  URL getResource(String resource);

  /**
   * Same as {@link #loadFrom(String, String, String, Optional, String, String)} but using no classifier and {@code jar} as
   * type.
   *
   * @param resource the name of the resource to load
   * @param groupId the group ID of the artifact where to look
   * @param artifactId the artifact ID of the artifact where to look
   * @param version the optional version of the artifact where to look
   * @return a stream for this resource
   */
  default Optional<InputStream> loadFrom(String resource, String groupId, String artifactId, Optional<String> version) {
    return loadFrom(resource, groupId, artifactId, version, "", "jar");
  }

  /**
   * Looks for a resource as in {@link #findIn(String, String, String, Optional, String, String)} and returns it's {@link InputStream} if
   * possible.
   *
   * @param resource the name of the resource to load
   * @param groupId the group ID of the artifact where to look
   * @param artifactId the artifact ID of the artifact where to look
   * @param version the optional version of the artifact where to look
   * @param classifier the classifier of the artifact where to look
   * @param type the type of the artifact where to look
   * @return a stream for this resource
   */
  Optional<InputStream> loadFrom(String resource, String groupId, String artifactId, Optional<String> version, String classifier,
                                 String type);

  /**
   * Looks for a resource based on the calling class classloader and the current context classloader, specifically within the artifact
   * specified by it's GAV coordinates, considering it's classifier and type as well.
   *
   * @param resource the name of the resource to load
   * @param groupId the group ID of the artifact where to look
   * @param artifactId the artifact ID of the artifact where to look
   * @param version the optional version of the artifact where to look
   * @param classifier the classifier of the artifact where to look
   * @param type the type of the artifact where to look
   * @return the {@link URL} for this resource
   */
  Optional<URL> findIn(String resource, String groupId, String artifactId, Optional<String> version, String classifier,
                       String type);

  /**
   * Same as {@link #findIn(String, String, String, Optional, String, String)} but using no classifier and {@code jar} as
   * type.
   *
   * @param resource the name of the resource to load
   * @param groupId the group ID of the artifact where to look
   * @param artifactId the artifact ID of the artifact where to look
   * @param version the optional version of the artifact where to look
   * @return the {@link URL} for this resource
   */
  default Optional<URL> findIn(String resource, String groupId, String artifactId, Optional<String> version) {
    return findIn(resource, groupId, artifactId, version, "", "jar");
  }

}
