/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */

package org.mule.tooling.client.api.exception;

import static java.lang.System.lineSeparator;
import static java.util.Optional.empty;
import static java.util.Optional.of;

import java.util.Optional;

import org.apache.commons.lang3.exception.ExceptionUtils;

/**
 * @since 1.0
 */
public class ToolingException extends RuntimeException {

  private String rootCauseMessage;
  private String rootCauseType;
  private String[] rootCauseStackTrace;

  public ToolingException(String message) {
    super(message);
  }

  public ToolingException(Throwable cause) {
    super(buildMessage(empty(), ExceptionUtils.getStackTrace(cause)), null);
    setRootCauseMessage(cause);
    setRootCauseType(cause);
    setRootCauseStackTrace(cause);
  }

  public ToolingException(String message, Throwable cause) {
    super(buildMessage(of(message), ExceptionUtils.getStackTrace(cause)), null);
    setRootCauseMessage(cause);
    setRootCauseType(cause);
    setRootCauseStackTrace(cause);
  }

  private static String buildMessage(Optional<String> message, String cause) {
    final StringBuilder stringBuilder = new StringBuilder();
    message.ifPresent(stringBuilder::append);
    stringBuilder.append(lineSeparator());
    stringBuilder.append(cause);
    return stringBuilder.toString();
  }

  private void setRootCauseMessage(Throwable cause) {
    rootCauseMessage = ExceptionUtils.getRootCauseMessage(cause);
  }

  private void setRootCauseType(Throwable cause) {
    Throwable rootCause = ExceptionUtils.getRootCause(cause);
    rootCause = rootCause == null ? cause : rootCause;
    rootCauseType = rootCause.getClass().getName();
  }

  private void setRootCauseStackTrace(Throwable cause) {
    rootCauseStackTrace = ExceptionUtils.getRootCauseStackTrace(cause);
  }

  public String getRootCauseMessage() {
    return rootCauseMessage;
  }

  public String getRootCauseType() {
    return rootCauseType;
  }

  public String[] getRootCauseStackTrace() {
    return rootCauseStackTrace;
  }
}
