/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */

package org.mule.tooling.client.internal.serialization;

import static org.hamcrest.Matchers.equalTo;
import static org.hamcrest.Matchers.is;
import static org.junit.Assert.assertThat;
import static org.mule.metadata.api.model.MetadataFormat.JAVA;
import org.mule.metadata.api.builder.ArrayTypeBuilder;
import org.mule.metadata.api.builder.BaseTypeBuilder;
import org.mule.metadata.api.builder.ObjectTypeBuilder;
import org.mule.metadata.api.model.ArrayType;
import org.mule.metadata.api.model.StringType;
import org.mule.metadata.internal.utils.MetadataTypeWriter;
import org.mule.metadata.java.api.annotation.ClassInformationAnnotation;

import java.util.List;

import org.junit.Test;

public abstract class DefaultArrayTypeSerializerTestCase {

  private static final String DESCRIPTION = "description";

  @Test
  public void keepAnnotationsAtArrayTypeLevel() {
    StringType stringType = new BaseTypeBuilder(JAVA)
        .stringType()
        .with(new ClassInformationAnnotation(String.class))
        .build();

    ArrayType arrayType = new BaseTypeBuilder(JAVA)
        .arrayType()
        .description(DESCRIPTION)
        .of(stringType)
        .with(new ClassInformationAnnotation(List.class))
        .build();

    Serializer serializer = createSerializer();
    String content = serializer.serialize(arrayType);
    ArrayType deserialized = serializer.deserialize(content);

    assertThat(deserialized.getAnnotation(ClassInformationAnnotation.class).isPresent(), is(true));
  }

  @Test
  public void recursiveReferences() {
    ArrayTypeBuilder arrayTypeBuilder = new BaseTypeBuilder(JAVA)
        .arrayType();

    ObjectTypeBuilder objectTypeBuilder = new BaseTypeBuilder(JAVA)
        .objectType();
    objectTypeBuilder.addField().key("NK1-17").value(arrayTypeBuilder);

    ObjectTypeBuilder parentObjectTypeBuilder = new BaseTypeBuilder(JAVA)
        .objectType();

    parentObjectTypeBuilder.addField().value(objectTypeBuilder);

    arrayTypeBuilder.of(objectTypeBuilder);

    ArrayType arrayType = arrayTypeBuilder.build();

    Serializer serializer = createSerializer();
    String content = serializer.serialize(arrayType);
    ArrayType deserialized = serializer.deserialize(content);

    assertThat(new MetadataTypeWriter().toString(arrayType), equalTo(new MetadataTypeWriter().toString(deserialized)));
  }

  protected abstract Serializer createSerializer();
}
