/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */

package org.mule.tooling.client.api.datasense;

import static java.lang.String.format;
import static java.util.Objects.requireNonNull;
import static org.apache.commons.lang3.ClassUtils.getShortClassName;
import org.mule.tooling.client.api.artifact.ToolingArtifact;
import org.mule.tooling.client.api.component.location.Location;
import org.mule.tooling.client.api.request.AbstractToolingRequest;

/**
 * Request for {@link DataSenseService}.
 *
 * @since 1.0
 */
public class DataSenseRequest extends AbstractToolingRequest {

  private long metadataTimeout = -1;

  private DataSenseResolutionScope dataSenseResolutionScope;

  /**
   * Sets the location from the {@link ToolingArtifact} configuration to be used for resolving DataSense.
   *
   * @param location from the {@link ToolingArtifact} configuration.
   */
  public void setLocation(Location location) {
    requireNonNull(location, "location cannot null");
    setDataSenseResolutionScope(new ComponentResolutionScope(location));
  }

  /**
   * @return the componentPath from the {@link ToolingArtifact} configuration.
   */
  public Location getLocation() {
    final DataSenseResolutionScope dataSenseResolutionScope = getDataSenseResolutionScope();
    if (dataSenseResolutionScope instanceof ComponentResolutionScope) {
      return ((ComponentResolutionScope) dataSenseResolutionScope).getLocation();
    } else {
      return null;
    }
  }

  /**
   * Sets the resolution scope to be used on resolving DataSense.
   *
   * @param dataSenseResolutionScope resolution scope to use.
   */
  public void setDataSenseResolutionScope(DataSenseResolutionScope dataSenseResolutionScope) {
    requireNonNull(dataSenseResolutionScope, "dataSenseResolutionScope cannot null");
    this.dataSenseResolutionScope = dataSenseResolutionScope;
  }

  /**
   * @return the {@link DataSenseResolutionScope}.
   */
  public DataSenseResolutionScope getDataSenseResolutionScope() {
    return dataSenseResolutionScope;
  }

  /**
   * Sets the timeout for metadata requests to be used for resolving DataSense.
   *
   * @param timeout from the {@link ToolingArtifact} configuration.
   */
  public void setMetadataTimeout(long timeout) {
    this.metadataTimeout = timeout;
  }

  /**
   * @return the timeout for metadata requests.
   */
  public long getMetadataTimeout() {
    return metadataTimeout;
  }

  @Override
  public String toString() {
    return format("%s{requestTimeout=%s,metadataRequestTimeout=%s,location=%s}", getShortClassName(this.getClass()),
                  getRequestTimeout(),
                  getMetadataTimeout(),
                  getLocation());
  }

}
