/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.tooling.client.api.cache;

import java.util.Optional;
import java.util.Set;

/**
 * Defines the storage mechanism used to build a Cache.
 * The cache will be used to store resolved values from a request to a service. (e.g: Metadata, ValueProvider, etc) and avoid computing it multiple times.
 * </p>
 * In case an specific implementation needs to serialize the data prior to storage, they should keep a reference of the
 * {@link CacheStorageSerializer}s available in the {@link CacheStorageFactory} when executing
 * {@link CacheStorageFactory#createStorage(CacheStorageFactoryContext)};
 *
 * In case serializers are used, implementations should follow this rule:
 *
 * If a value was serialized to be stored {@link CacheStorage#put(String, Object)},
 * it should be deserialized when returned {@link CacheStorage#get(String)}
 * That means that, weather or not the value was serialized, the following should always be true:
 * <code>
 *    storage.put("KEY", object);
 *    Object retrievedObject = storage.get("KEY").get();
 *    assert object.getClass().isAssignableFrom(retrievedObject.class) == true
 * </code>
 *
 * @since 1.3.0
 */
public interface CacheStorage {

  /**
   * Get the value from the cache associated with the key {@param key}
   *
   * @param key the key used to identify the value in the storage
   * @return the an {@link Optional} with the value stored or {@link Optional#empty()} if none found.
   * */
  <V> Optional<V> get(String key);

  /**
   * Add the value related with this key to the cache
   * @param key the key to use to locate the value
   * @param value the value to be stored
   * @return the value stored
   */
  <V> V put(String key, V value);

  /**
   * Remove the values associated with the keys provided from the cache
   * @param keys the keys to locate the values to remove
   */
  void remove(String... keys);

  /**
   * @return a {@link Set <String>} with all the keys being used in the storage.
   */
  Set<String> getAllKeys();

}
