/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */

package org.mule.tooling.client.api.metadata;

import org.mule.tooling.client.api.component.location.Location;
import org.mule.tooling.client.api.exception.ServiceUnavailableException;
import org.mule.tooling.client.api.exception.ToolingException;
import org.mule.tooling.client.api.feature.Action;
import org.mule.tooling.client.api.feature.Feature;

/**
 * Service for retrieving metadata.
 *
 * A {@code MetadataService}
 *
 * @since 1.0
 */
public interface MetadataService {

  /**
   * Returns a {@link MetadataResult} of {@link MetadataKeysContainer} described by {@link
   * MetadataKeysContainer#getKeys(String)}} of the associated
   * MetadataKeyProvider component identified by the given {@link MetadataKeysRequest}.
   *
   * @param metadataKeysRequest request to get the {@link MetadataKeysContainer}
   * @return Successful {@link MetadataResult} if the keys are successfully resolved Failure {@link MetadataResult} if there is an error while resolving the keys.
   * @throws ServiceUnavailableException if a connection exception happens when trying to connect to Mule Agent REST API.
   * @throws ToolingException if an unexpected error happens.
   */
  MetadataResult<MetadataKeysContainer> getMetadataKeys(MetadataKeysRequest metadataKeysRequest)
      throws ServiceUnavailableException;

  /**
   * If supported by the {@link org.mule.tooling.client.api.ToolingRuntimeClient} implementation will return a {@link Feature<Action>}
   * to do the dispose of Metadata on both, the Mule Runtime side cache and the Tooling Client side cache, for the given location.
   * </p>
   * This should be called when external changes on remote system are made and metadata has to be recalculated from scratch.
   * The location could be at component level or configuration, in any case all metadata resources in cache associated to the configuration referenced
   * by the component will be disposed.
   * If the {@link Location} corresponds to a configuration, then all metadata from all operations that reference that configuration will be invalidated.
   *
   * @return {@link Feature<Action>} to execute an {@link Action} for the location of the component to be cleared, could be a configuration or a component.
   *         Either one or the other all the metadata cache entries associated to the configuration would be disposed.
   */
  Feature<Action<Location>> disposeMetadataCache();

  /**
   * If supported by the {@link org.mule.tooling.client.api.ToolingRuntimeClient), the implementation will return a {@link Feature<Action<Location>>}}
   * to invalidate entries related to metadata keys calculated for the component associated with the {@link Location} given. This includes
   * disposing all resources cached in the Mule Runtime cache and the metadata keys stored in the Tooling client's metadata cache.
   * <p/>
   * This should be called when external changes on remote system are made and metadata keys have to be recalculated from scratch.
   * <p/>
   * Unlike {@link MetadataService#disposeMetadataCache()}, where if the {@link Location} corresponds to a configuration, then all related operations are
   * invalidated, this method only acts on the component associated with the given {@link Location}
   *
   * @return a {@link Feature<Action<Location>> } that invalidates all metadata keys from the component related to the given {@link Location} if enabled.
   *
   * @since 4.2.2, 4.3.0
   */
  Feature<Action<Location>> invalidateMetadataCacheKeys();

}
