/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */

package org.mule.tooling.client.api.artifact;

import static java.util.Optional.empty;

import org.mule.tooling.client.api.Disposable;
import org.mule.tooling.client.api.artifact.ast.ArtifactAst;
import org.mule.tooling.client.api.artifact.resources.ResourceLoader;
import org.mule.tooling.client.api.component.location.ComponentLocationService;
import org.mule.tooling.client.api.connectivity.ConnectivityTestingService;
import org.mule.tooling.client.api.datasense.DataSenseService;
import org.mule.tooling.client.api.dataweave.DataWeaveService;
import org.mule.tooling.client.api.feature.Feature;
import org.mule.tooling.client.api.metadata.MetadataService;
import org.mule.tooling.client.api.sampledata.SampleDataService;
import org.mule.tooling.client.api.value.provider.ValueProviderService;

import java.util.Map;
import java.util.Optional;

/**
 * Provides tooling operations for a Mule Artifact.
 * <p/>
 * Be aware that this class is not thread safe, due to it keeps state of the application deployed on Mule Runtime and all the
 * services that rely on it cannot be invoked in parallel as they use the same application deployed on Mule Runtime side and the
 * application is lazily initialized and cannot handle requests in parallel. The only scenario where the same
 * {@link ToolingArtifact} could be used in parallel is whenever resolving MetadataKeys or ValueProviders for the same component
 * location.
 *
 * If there is a case where clients need to resolve different operations in parallel it is recommended to work with multiple
 * instances of this class for the same application content. Please consider that a {@link ToolingArtifact} also has in memory
 * resources such as the application {@link ClassLoader}, {@link org.mule.tooling.client.api.extension.model.ExtensionModel
 * ExtensionModels}.
 *
 * @since 1.0
 */
public interface ToolingArtifact extends Disposable {

  /**
   * @return an the identifier for this {@link ToolingArtifact} in order to allow be fetched later.
   */
  String getId();

  /**
   * @return {@link Map} of properties defined for this {@link ToolingArtifact}.
   *
   * @since 1.1
   */
  Map<String, String> getProperties();

  /**
   * @return {@link ToolingArtifact} representing the parent, as a {@link ToolingArtifact} represents a Mule domain or application
   *         on those cases where this {@link ToolingArtifact} represents a Mule domain no parent will be set or if the
   *         application has not declared a domain dependency (default one will be used). In those cases where the
   *         {@link ToolingArtifact} has been created for a Mule application and set the reference to the parent
   *         {@link ToolingArtifact} that value will be returned here.
   *         <p/>
   *         In addition to that if the {@link ToolingArtifact} has been created without the parent reference but declares a
   *         domain dependency a {@link ToolingArtifact} will be created for the domain discovered.
   */
  default Optional<ToolingArtifact> getParent() {
    return empty();
  }

  // TODO (gfernandes) MULE-13231: Spike - Tooling should allow reload ApplicationModel on configuration changes
  /// **
  // * Calling this method will invalidate the internal caches that {@link org.mule.tooling.client.api.ToolingRuntimeClient has
  // for this {@link ToolingArtifact}.
  // * This mean that if an application was deployed for resolving operations through the Mule Runtime on this session it will be
  // disposed in order to be deployed again
  // * whenever the {@link ToolingArtifact} requires to fetch data from Mule Runtime, also if the application {@link ClassLoader}
  // was created and cached it will be closed and disposed in order to generate a new one.
  // * <p/>
  // * Common scenario when this method should be called is whenever the user changes the {@code pom.xml} for the application. In
  // case of remote packaging
  // * this should be called whenever resources are changed too.
  // *
  // * @param {@link URL} with the application's updated content.
  // */
  // void refresh(URL applicationUrlContent);

  /// **
  // * Disposing a {@link ToolingArtifact} will release any resources that were created for the session, similar to {@link
  /// #refresh(URL)} but with the addition
  // * that the {@link ToolingArtifact} cannot be fetched anymore.
  // */
  // @Override
  // void dispose();

  /**
   * Returns a {@link ConnectivityTestingService} to allow testing connections over the {@link ToolingArtifact} configuration.
   *
   * @return a {@link ConnectivityTestingService}
   */
  ConnectivityTestingService connectivityTestingService();

  /**
   * Returns a {@link MetadataService} to allow retrieving metadata over the {@link ToolingArtifact} configuration elements.
   *
   * @return a {@link MetadataService}
   */
  MetadataService metadataService();

  /**
   * Returns a {@link DataSenseService} to resolve data sense operations over the {@link ToolingArtifact}.
   *
   * @return a {@link DataSenseService}
   */
  DataSenseService dataSenseService();

  /**
   * Returns a {@link DataWeaveService} to run a DataWeave script over an application context.
   *
   * @return a {@link DataWeaveService}
   */
  DataWeaveService dataWeaveService();

  /**
   * Returns a {@link ValueProviderService} to be able to resolve values for a given Value Provider.
   *
   * @return a {@link ValueProviderService}
   */
  ValueProviderService valueProviderService();

  /**
   * Returns a {@link ValueProviderService} to be able to resolve values for a given Value Provider.
   *
   * @return a {@link ValueProviderService}
   */
  Feature<SampleDataService> sampleDataService();

  /**
   * Provides a {@link ResourceLoader} that supports to access resources inside the artifact with the same class loading filtering
   * rules that Mule Runtime has.
   *
   * @return a {@link Feature} for the {@link ResourceLoader} to access resources inside the artifact.
   */
  Feature<ResourceLoader> getResourceLoader();

  ComponentLocationService componentLocationService();

  /**
   * Returns the {@link ArtifactAst} representation for the underlying artifact.
   *
   * @return the {@link ArtifactAst} representation for the underlying artifact.
   *
   * @since 1.4
   */
  ArtifactAst getArtifactAst();

}
