/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */

package org.mule.tooling.client.api.configuration.ssl;

import static java.util.Objects.requireNonNull;
import org.mule.tooling.client.internal.configuration.ssl.ImmutableSslConfiguration;

/**
 * Configuration for an {@link javax.net.ssl.SSLContext}.
 *
 * @since 1.0
 */
public interface SslConfiguration {

  /**
   * @return {@link KeyStoreConfig} for this configuration.
   */
  KeyStoreConfig getKeyStoreConfig();

  /**
   * @return {@link TrustStoreConfig} for this configuration.
   */
  TrustStoreConfig getTrustStoreConfig();

  /**
   * @return a {@link Builder} to build a {@link SslConfiguration}.
   */
  static Builder builder() {
    return new Builder();
  }

  class Builder {

    private KeyStoreConfig keyStoreConfig;
    private TrustStoreConfig trustStoreConfig;

    /**
     * Sets the key store config {@link KeyStoreConfig}.
     *
     * @param keyStoreConfig key store config {@link KeyStoreConfig}.
     * @return this
     */
    public Builder withKeyStoreConfig(KeyStoreConfig keyStoreConfig) {
      requireNonNull(keyStoreConfig, "keyStoreConfig cannot be null");
      this.keyStoreConfig = keyStoreConfig;
      return this;
    }

    /**
     * Sets trust store config {@link TrustStoreConfig}.
     *
     * @param trustStoreConfig trust store config {@link TrustStoreConfig}.
     * @return this
     */
    public Builder withTrustStoreConfig(TrustStoreConfig trustStoreConfig) {
      requireNonNull(trustStoreConfig, "trustStoreConfig cannot be null");
      this.trustStoreConfig = trustStoreConfig;
      return this;
    }

    /**
     * Builds the {@link SslConfiguration} object.
     *
     * @return {@link SslConfiguration} with the value sets.
     */
    public SslConfiguration build() {
      return new ImmutableSslConfiguration(this.keyStoreConfig, this.trustStoreConfig);
    }

  }

}
