/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */

package org.mule.tooling.client.api.descriptors;

import static java.lang.String.format;
import static java.util.Collections.emptyMap;
import static java.util.Collections.unmodifiableMap;
import static java.util.Objects.requireNonNull;
import static org.apache.commons.lang3.ClassUtils.getShortClassName;

import java.util.HashMap;
import java.util.Map;

/**
 * A descriptor for an artifact represented in Maven coordinates.
 *
 * @since 1.0
 */
public final class ArtifactDescriptor {

  private static final String JAR = "jar";

  private final String groupId;
  private final String artifactId;
  private final String version;
  private final String classifier;
  private final String extension;

  private Map<String, String> properties = emptyMap();

  /**
   * Builder for {@link ArtifactDescriptor Artifacts}.
   *
   * @since 1.0
   */
  public static class Builder {

    private String groupId;
    private String artifactId;
    private String extension;
    private String version;
    private String classifier;
    private Map<String, String> properties = new HashMap<>();

    public Builder withGroupId(String groupId) {
      this.groupId = groupId;
      return this;
    }

    public Builder withArtifactId(String artifactId) {
      this.artifactId = artifactId;
      return this;
    }

    public Builder withExtension(String extension) {
      this.extension = extension;
      return this;
    }

    public Builder withVersion(String version) {
      this.version = version;
      return this;
    }

    public Builder withClassifier(String classifier) {
      this.classifier = classifier;
      return this;
    }

    public Builder withProperties(Map<String, String> properties) {
      this.properties.putAll(properties);
      return this;
    }

    public ArtifactDescriptor build() {
      return new ArtifactDescriptor(groupId, artifactId, version, classifier, extension, properties);
    }
  }

  public static Builder newBuilder() {
    return new Builder();
  }

  /**
   * Creates an instance of an {@link ArtifactDescriptor}.
   *
   * @param groupId    Maven groupId. Not null.
   * @param artifactId Maven artifactId. Not null.
   * @param version    Maven version. Not null.
   * @param classifier Maven classifier.
   * @param extension  Maven file extension (jar, zip). If null jar will be used.
   * @param properties to allow setting dynamic configurations for those who use this artifact descriptor.
   */
  private ArtifactDescriptor(String groupId, String artifactId, String version, String classifier, String extension,
                             Map<String, String> properties) {
    requireNonNull(groupId, "groupId");
    requireNonNull(artifactId, "artifactId");
    requireNonNull(version, "version");

    this.groupId = groupId;
    this.artifactId = artifactId;
    this.version = version;
    this.classifier = classifier;
    this.extension = extension == null ? JAR : extension;
    this.properties = properties;
  }

  /**
   * Creates an instance of an {@link ArtifactDescriptor}.
   *
   * @param groupId    Maven groupId. Not null.
   * @param artifactId Maven artifactId. Not null.
   * @param version    Maven version. Not null.
   * @param classifier Maven classifier.
   * @param extension  Maven file extension (jar, zip). If null jar will be used.
   */
  private ArtifactDescriptor(String groupId, String artifactId, String version, String classifier, String extension) {
    this(groupId, artifactId, version, classifier, extension, emptyMap());
  }

  public String getGroupId() {
    return groupId;
  }

  public String getArtifactId() {
    return artifactId;
  }

  public String getVersion() {
    return version;
  }

  public String getExtension() {
    return extension;
  }

  public String getClassifier() {
    return classifier;
  }

  public Map<String, String> getProperties() {
    // We need to explicitly do this as previous version may be used and Kryo
    // will not call the initialization code.
    if (properties == null) {
      properties = emptyMap();
    }
    return unmodifiableMap(properties);
  }

  @Override
  public boolean equals(Object o) {
    if (this == o) {
      return true;
    }
    if (o == null || getClass() != o.getClass()) {
      return false;
    }

    ArtifactDescriptor artifactDescriptor = (ArtifactDescriptor) o;

    if (!groupId.equals(artifactDescriptor.groupId)) {
      return false;
    }
    if (!artifactId.equals(artifactDescriptor.artifactId)) {
      return false;
    }
    if (!version.equals(artifactDescriptor.version)) {
      return false;
    }
    return extension.equals(artifactDescriptor.extension);

  }

  @Override
  public int hashCode() {
    int result = groupId.hashCode();
    result = 31 * result + artifactId.hashCode();
    result = 31 * result + version.hashCode();
    result = 31 * result + extension.hashCode();
    return result;
  }

  @Override
  public String toString() {
    return format("%s{groupId=%s,artifactId=%s,version=%s,extension=%s,properties=%s}", getShortClassName(this.getClass()),
                  getGroupId(),
                  getArtifactId(), getVersion(), getExtension(), getProperties());
  }

}
