/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */

package org.mule.tooling.client.api.extension.model.connection;

import static java.util.Collections.emptySet;
import static java.util.Collections.unmodifiableSet;
import static java.util.Optional.ofNullable;
import static org.mule.tooling.client.api.feature.Feature.disabled;
import static org.mule.tooling.client.api.feature.Feature.enabled;

import org.mule.tooling.client.api.extension.model.DisplayModel;
import org.mule.tooling.client.api.extension.model.ExternalLibraryModel;
import org.mule.tooling.client.api.extension.model.ParameterizedModel;
import org.mule.tooling.client.api.extension.model.StereotypeModel;
import org.mule.tooling.client.api.extension.model.deprecated.DeprecationModel;
import org.mule.tooling.client.api.extension.model.parameter.ParameterGroupModel;
import org.mule.tooling.client.api.extension.model.semantic.HasSemanticTerms;
import org.mule.tooling.client.api.feature.Feature;

import java.util.HashSet;
import java.util.List;
import java.util.Optional;
import java.util.Set;

import org.apache.commons.lang3.builder.ToStringBuilder;

/**
 * Introspection model for ConnectionProvider types.
 * <p>
 * Provider models implement the flyweight pattern. This means that a given operation should only be represented by only one
 * instance of this class. Thus, if the same operation is contained by different instances, then each of those containers should
 * reference the same operation model instance.
 *
 * @since 1.0
 */
public class ConnectionProviderModel implements ParameterizedModel, HasSemanticTerms {

  private String name;
  private String description;
  private DisplayModel displayModel;
  private ConnectionManagementType connectionManagementType;
  private List<ParameterGroupModel> parameterGroupModels;
  private Set<ExternalLibraryModel> externalLibraryModels;
  private StereotypeModel stereotype;
  private boolean supportsConnectivityTesting = true;
  private Feature<DeprecationModel> deprecationModel;

  // There is no point in making this a feature. If disabled, we can just return an empty set
  private Set<String> semanticTerms;

  // Just needed in order to serialize this object
  private ConnectionProviderModel() {}

  public ConnectionProviderModel(String name, String description,
                                 DisplayModel displayModel,
                                 ConnectionManagementType connectionManagementType,
                                 boolean supportsConnectivityTesting,
                                 List<ParameterGroupModel> parameterGroupModels,
                                 Set<ExternalLibraryModel> externalLibraryModels,
                                 StereotypeModel stereotype,
                                 DeprecationModel deprecationModel) {
    this(name, description, displayModel, connectionManagementType, supportsConnectivityTesting, parameterGroupModels,
         externalLibraryModels, stereotype, deprecationModel, emptySet());
  }

  public ConnectionProviderModel(String name, String description,
                                 DisplayModel displayModel,
                                 ConnectionManagementType connectionManagementType,
                                 boolean supportsConnectivityTesting,
                                 List<ParameterGroupModel> parameterGroupModels,
                                 Set<ExternalLibraryModel> externalLibraryModels,
                                 StereotypeModel stereotype,
                                 DeprecationModel deprecationModel,
                                 Set<String> semanticTerms) {
    this.name = name;
    this.description = description;
    this.displayModel = displayModel;
    this.connectionManagementType = connectionManagementType;
    this.supportsConnectivityTesting = supportsConnectivityTesting;
    this.parameterGroupModels = parameterGroupModels;
    this.externalLibraryModels = externalLibraryModels;
    this.stereotype = stereotype;
    this.deprecationModel = enabled(deprecationModel);
    this.semanticTerms = new HashSet<>(semanticTerms);
  }

  public String getName() {
    return name;
  }

  public String getDescription() {
    return description;
  }

  public Optional<DisplayModel> getDisplayModel() {
    return ofNullable(displayModel);
  }

  public ConnectionManagementType getConnectionManagementType() {
    return connectionManagementType;
  }

  @Override
  public List<ParameterGroupModel> getParameterGroupModels() {
    return parameterGroupModels;
  }

  public Set<ExternalLibraryModel> getExternalLibraryModels() {
    return externalLibraryModels;
  }

  public StereotypeModel getStereotype() {
    return stereotype;
  }

  public boolean supportsConnectivityTesting() {
    return supportsConnectivityTesting;
  }

  public Feature<DeprecationModel> getDeprecationModel() {
    if (deprecationModel == null) {
      deprecationModel = disabled();
    }
    return deprecationModel;
  }

  // Helper methods

  @Override
  public Optional<ParameterGroupModel> getParameterGroupModel(String name) {
    return parameterGroupModels.stream().filter(model -> model.getName().equals(name)).findFirst();
  }

  @Override
  public Set<String> getSemanticTerms() {
    if (semanticTerms == null) {
      this.semanticTerms = emptySet();
    }
    return unmodifiableSet(semanticTerms);
  }

  @Override
  public int hashCode() {
    return this.name.hashCode();
  }

  @Override
  public boolean equals(Object obj) {
    return this.getClass().isInstance(obj) && this.name.equals(((ConnectionProviderModel) obj).getName());
  }

  @Override
  public String toString() {
    return ToStringBuilder.reflectionToString(this);
  }

}
