/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */

package org.mule.tooling.client.api.extension.model.source;

import static java.util.Collections.emptySet;
import static java.util.Collections.unmodifiableSet;
import static java.util.Optional.ofNullable;
import static org.mule.tooling.client.api.feature.Feature.disabled;
import static org.mule.tooling.client.api.feature.Feature.enabled;
import org.mule.tooling.client.api.extension.model.ComponentModel;
import org.mule.tooling.client.api.extension.model.DisplayModel;
import org.mule.tooling.client.api.extension.model.ErrorModel;
import org.mule.tooling.client.api.extension.model.ExtensionModel;
import org.mule.tooling.client.api.extension.model.OutputModel;
import org.mule.tooling.client.api.extension.model.StereotypeModel;
import org.mule.tooling.client.api.extension.model.data.sample.SampleDataProviderModel;
import org.mule.tooling.client.api.extension.model.deprecated.DeprecationModel;
import org.mule.tooling.client.api.extension.model.metadata.MetadataKeyIdModel;
import org.mule.tooling.client.api.extension.model.metadata.TypeResolversInformationModel;
import org.mule.tooling.client.api.extension.model.nested.NestableElementModel;
import org.mule.tooling.client.api.extension.model.parameter.ParameterGroupModel;
import org.mule.tooling.client.api.feature.Feature;

import java.util.HashSet;
import java.util.List;
import java.util.Optional;
import java.util.Set;

import org.apache.commons.lang3.builder.ToStringBuilder;

/**
 * A definition of a message source in an {@link ExtensionModel}.
 *
 * @since 1.0
 */
public class SourceModel implements ComponentModel {

  private String name;
  private String description;
  private List<ParameterGroupModel> parameterGroupModels;
  private DisplayModel displayModel;
  private OutputModel output;
  private OutputModel outputAttributes;
  private boolean transactional;
  private boolean requiresConnection;
  private boolean supportsStreaming;
  private StereotypeModel stereotype;
  private boolean hasResponse;
  private SourceCallbackModel successCallback;
  private SourceCallbackModel errorCallback;
  private SourceCallbackModel terminateCallbackModel;
  private Set<ErrorModel> errors;
  private List<? extends NestableElementModel> nestedComponents;
  private Feature<Boolean> runsOnPrimaryNodeOnly;
  private Feature<DeprecationModel> deprecationModel;
  private Feature<SampleDataProviderModel> sampleDataProviderModel;

  // There is no point in making this a feature. If disabled, we can just return an empty set
  private Set<String> semanticTerms;

  // Model Properties
  private MetadataKeyIdModel metadataKeyIdModel;
  private Feature<TypeResolversInformationModel> typeResolversInformationModel;

  // Just needed in order to serialize this object
  private SourceModel() {}

  @Deprecated
  public SourceModel(String name, String description,
                     List<ParameterGroupModel> parameterGroupModels,
                     DisplayModel displayModel, OutputModel output,
                     OutputModel outputAttributes, boolean transactional, boolean requiresConnection,
                     boolean supportsStreaming,
                     StereotypeModel stereotype,
                     boolean hasResponse,
                     boolean runsOnPrimaryNodeOnly,
                     SourceCallbackModel successCallback,
                     SourceCallbackModel errorCallback,
                     SourceCallbackModel terminateCallbackModel,
                     Set<ErrorModel> errors,
                     List<? extends NestableElementModel> nestedComponents,
                     MetadataKeyIdModel metadataKeyIdModel,
                     DeprecationModel deprecationModel) {
    this(name, description, parameterGroupModels, displayModel, output, outputAttributes, transactional,
         requiresConnection, supportsStreaming, stereotype, hasResponse, runsOnPrimaryNodeOnly, successCallback, errorCallback,
         terminateCallbackModel, errors, nestedComponents, metadataKeyIdModel, deprecationModel,
         null, null);
  }

  public SourceModel(String name, String description,
                     List<ParameterGroupModel> parameterGroupModels,
                     DisplayModel displayModel, OutputModel output,
                     OutputModel outputAttributes, boolean transactional, boolean requiresConnection,
                     boolean supportsStreaming,
                     StereotypeModel stereotype,
                     boolean hasResponse,
                     boolean runsOnPrimaryNodeOnly,
                     SourceCallbackModel successCallback,
                     SourceCallbackModel errorCallback,
                     SourceCallbackModel terminateCallbackModel,
                     Set<ErrorModel> errors,
                     List<? extends NestableElementModel> nestedComponents,
                     MetadataKeyIdModel metadataKeyIdModel,
                     DeprecationModel deprecationModel,
                     TypeResolversInformationModel typeResolversInformationModel,
                     SampleDataProviderModel sampleDataProviderModel) {
    this(name, description, parameterGroupModels, displayModel, output, outputAttributes, transactional,
         requiresConnection, supportsStreaming, stereotype, hasResponse, runsOnPrimaryNodeOnly, successCallback, errorCallback,
         terminateCallbackModel, errors, nestedComponents, metadataKeyIdModel, deprecationModel,
         typeResolversInformationModel, sampleDataProviderModel, emptySet());
  }

  public SourceModel(String name, String description,
                     List<ParameterGroupModel> parameterGroupModels,
                     DisplayModel displayModel, OutputModel output,
                     OutputModel outputAttributes, boolean transactional, boolean requiresConnection,
                     boolean supportsStreaming,
                     StereotypeModel stereotype,
                     boolean hasResponse,
                     boolean runsOnPrimaryNodeOnly,
                     SourceCallbackModel successCallback,
                     SourceCallbackModel errorCallback,
                     SourceCallbackModel terminateCallbackModel,
                     Set<ErrorModel> errors,
                     List<? extends NestableElementModel> nestedComponents,
                     MetadataKeyIdModel metadataKeyIdModel,
                     DeprecationModel deprecationModel,
                     TypeResolversInformationModel typeResolversInformationModel,
                     SampleDataProviderModel sampleDataProviderModel,
                     Set<String> semanticTerms) {
    this.name = name;
    this.description = description;
    this.parameterGroupModels = parameterGroupModels;
    this.displayModel = displayModel;
    this.output = output;
    this.outputAttributes = outputAttributes;
    this.transactional = transactional;
    this.requiresConnection = requiresConnection;
    this.supportsStreaming = supportsStreaming;
    this.stereotype = stereotype;
    this.hasResponse = hasResponse;
    this.successCallback = successCallback;
    this.errorCallback = errorCallback;
    this.terminateCallbackModel = terminateCallbackModel;
    this.errors = errors;
    this.nestedComponents = nestedComponents;
    this.metadataKeyIdModel = metadataKeyIdModel;
    this.runsOnPrimaryNodeOnly = enabled(runsOnPrimaryNodeOnly);
    this.deprecationModel = enabled(deprecationModel);
    this.typeResolversInformationModel = enabled(typeResolversInformationModel);
    this.sampleDataProviderModel = enabled(sampleDataProviderModel);
    this.semanticTerms = new HashSet<>(semanticTerms);
  }

  public String getName() {
    return name;
  }

  public String getDescription() {
    return description;
  }

  public List<ParameterGroupModel> getParameterGroupModels() {
    return parameterGroupModels;
  }

  public Optional<DisplayModel> getDisplayModel() {
    return ofNullable(displayModel);
  }

  public OutputModel getOutput() {
    return output;
  }

  public OutputModel getOutputAttributes() {
    return outputAttributes;
  }

  public boolean isTransactional() {
    return transactional;
  }

  public boolean requiresConnection() {
    return requiresConnection;
  }

  public boolean supportsStreaming() {
    return supportsStreaming;
  }

  public StereotypeModel getStereotype() {
    return stereotype;
  }

  public boolean hasResponse() {
    return hasResponse;
  }

  public Optional<SourceCallbackModel> getSuccessCallback() {
    return ofNullable(successCallback);
  }

  public Optional<SourceCallbackModel> getErrorCallback() {
    return ofNullable(errorCallback);
  }

  public Optional<SourceCallbackModel> getTerminateCallback() {
    return ofNullable(terminateCallbackModel);
  }

  public Set<ErrorModel> getErrorModels() {
    return errors;
  }

  public List<? extends NestableElementModel> getNestedComponents() {
    return nestedComponents;
  }

  public Optional<MetadataKeyIdModel> getMetadataKeyIdModel() {
    return ofNullable(metadataKeyIdModel);
  }

  public Feature<Boolean> getRunsOnPrimaryNodeOnly() {
    if (runsOnPrimaryNodeOnly == null) {
      runsOnPrimaryNodeOnly = disabled();
    }
    return runsOnPrimaryNodeOnly;
  }

  public Feature<DeprecationModel> getDeprecationModel() {
    if (deprecationModel == null) {
      deprecationModel = disabled();
    }
    return deprecationModel;
  }

  public Feature<TypeResolversInformationModel> getTypeResolversInformationModel() {
    if (typeResolversInformationModel == null) {
      typeResolversInformationModel = disabled();
    }
    return typeResolversInformationModel;
  }

  public Feature<SampleDataProviderModel> getSampleDataProviderModel() {
    if (sampleDataProviderModel == null) {
      sampleDataProviderModel = disabled();
    }
    return sampleDataProviderModel;
  }

  @Override
  public Set<String> getSemanticTerms() {
    if (semanticTerms == null) {
      this.semanticTerms = emptySet();
    }
    return unmodifiableSet(semanticTerms);
  }

  // Helper methods

  public Optional<ParameterGroupModel> getParameterGroupModel(String name) {
    return parameterGroupModels.stream().filter(model -> model.getName().equals(name)).findFirst();
  }

  @Override
  public int hashCode() {
    return this.name.hashCode();
  }

  @Override
  public boolean equals(Object obj) {
    return this.getClass().isInstance(obj) && this.name.equals(((SourceModel) obj).getName());
  }

  @Override
  public String toString() {
    return ToStringBuilder.reflectionToString(this);
  }

}
