/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.tooling.client.api.extension.model.value;

import static org.mule.tooling.client.internal.util.Preconditions.checkNotNull;

import org.mule.tooling.client.api.extension.model.parameter.ActingParameterModel;
import org.mule.tooling.client.api.extension.model.parameter.ParameterModel;

import java.util.List;

import org.apache.commons.lang3.builder.EqualsBuilder;
import org.apache.commons.lang3.builder.HashCodeBuilder;

/**
 * Model for {@link ParameterModel} to communicate if an {@link org.mule.metadata.api.model.ObjectFieldType} is capable to have
 * values.
 * <p>
 * The {@link org.mule.metadata.api.model.ObjectFieldType} defined for the type of this model will be considered as one that has
 * values to be prompted to users.
 *
 * @since 1.4
 */
public class FieldValuesResolverModel extends ValuesResolverModel {

  private final String targetSelector;

  /**
   * Creates a new instance
   *
   * @param targetSelector        the path to the field which is enriched with values
   * @param parameters            the list of parameters that the Values Resolver takes into account for its resolution
   * @param requiresConfiguration indicates if the configuration is required to resolve the values
   * @param requiresConnection    indicates if the connection is required to resolve the values
   * @param isOpen                indicates if the calculated values should be considered as an open or closed set
   * @param resolverName          resolver name to be used to resolve the values
   */
  public FieldValuesResolverModel(String targetSelector,
                                  List<ActingParameterModel> parameters,
                                  boolean requiresConfiguration,
                                  boolean requiresConnection,
                                  boolean isOpen,
                                  String resolverName) {
    super(parameters, requiresConfiguration, requiresConnection, isOpen, resolverName);
    checkNotNull(targetSelector, "'targetSelector' can't be null");
    this.targetSelector = targetSelector;
  }

  // For serialization
  private FieldValuesResolverModel() {
    this.targetSelector = null;
  }

  /**
   * @return the path to the field which is enriched with values
   */
  public String getTargetSelector() {
    return targetSelector;
  }

  @Override
  public boolean equals(Object o) {
    if (this == o) {
      return true;
    }

    if (o == null || getClass() != o.getClass()) {
      return false;
    }

    FieldValuesResolverModel that = (FieldValuesResolverModel) o;

    return new EqualsBuilder()
        .append(targetSelector, that.targetSelector)
        .isEquals() && super.equals(o);
  }

  @Override
  public int hashCode() {
    return new HashCodeBuilder(17, 37)
        .append(super.hashCode())
        .append(targetSelector)
        .toHashCode();
  }

  @Override
  public String toString() {
    return "ValuesResolverModel{" +
        "resolverName='" + getResolverName() + '\'' +
        ", requiresConfiguration=" + requiresConfiguration() +
        ", requiresConnection=" + requiresConnection() +
        ", isOpen=" + isOpen() +
        ", targetSelector='" + getTargetSelector() + '\'' +
        ", parameters=" + getParameters() +
        '}';
  }
}
