/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.tooling.client.api.extension.model.value;

import static org.mule.tooling.client.api.feature.Feature.disabled;
import static org.mule.tooling.client.api.feature.Feature.enabled;
import static org.mule.tooling.client.internal.util.Preconditions.checkNotNull;

import org.mule.api.annotation.NoInstantiate;
import org.mule.tooling.client.api.extension.model.parameter.ActingParameterModel;
import org.mule.tooling.client.api.extension.model.parameter.ParameterModel;
import org.mule.tooling.client.api.feature.Feature;

import java.util.List;

import org.apache.commons.lang3.builder.EqualsBuilder;
import org.apache.commons.lang3.builder.HashCodeBuilder;

/**
 * Model for {@link ParameterModel} to communicate if one of these are capable to provide
 * {@link org.mule.runtime.extension.api.dsql.Value values}.
 * <p>
 * The element with this model will considered as a one that provides values.
 *
 * @since 1.0
 */
@NoInstantiate
public class ValueProviderModel {

  private List<String> actingParameters;
  private Feature<List<ActingParameterModel>> parameters;
  private Integer partOrder;
  private String providerName;
  private Feature<String> providerId;
  private boolean requiresConfiguration;
  private boolean requiresConnection;
  private boolean isOpen;

  protected ValueProviderModel() {}

  /**
   * Creates a new instance
   *
   * @param actingParameters      the list of parameters that are required to execute the Value Provider resolution
   * @param requiresConfiguration indicates if the configuration is required to resolve the values
   * @param requiresConnection    indicates if the connection is required to resolve the values
   * @param isOpen                indicates if the calculated values should be considered as an open or closed set
   * @param partOrder             the position in the value
   * @param providerName          the category of the associated value provider for this parameter
   */
  public ValueProviderModel(List<String> actingParameters, boolean requiresConfiguration,
                            boolean requiresConnection,
                            boolean isOpen,
                            Integer partOrder, String providerName, String providerId) {
    checkNotNull(actingParameters, "'actingParameters' can't be null");
    checkNotNull(partOrder, "'valueParts' can't be null");
    checkNotNull(providerName, "'providerName' can't be null");
    this.isOpen = isOpen;
    this.actingParameters = actingParameters;
    this.parameters = disabled();
    this.requiresConfiguration = requiresConfiguration;
    this.requiresConnection = requiresConnection;
    this.partOrder = partOrder;
    this.providerName = providerName;
    this.providerId = enabled(providerId);
  }

  /**
   * Creates a new instance
   *
   * @param actingParameters      the list of parameters that are required to execute the Value Provider resolution
   * @param parameters            the list of parameters that the Value Provider takes into account for its resolution
   * @param requiresConfiguration indicates if the configuration is required to resolve the values
   * @param requiresConnection    indicates if the connection is required to resolve the values
   * @param isOpen                indicates if the calculated values should be considered as an open or closed set
   * @param partOrder             the position in the value
   * @param providerName          the category of the associated value provider for this parameter
   * @since 1.4
   */
  public ValueProviderModel(List<String> actingParameters, List<ActingParameterModel> parameters, boolean requiresConfiguration,
                            boolean requiresConnection,
                            boolean isOpen,
                            Integer partOrder, String providerName, String providerId) {
    checkNotNull(actingParameters, "'actingParameters' can't be null");
    checkNotNull(parameters, "'parameters' can't be null");
    checkNotNull(partOrder, "'valueParts' can't be null");
    checkNotNull(providerName, "'providerName' can't be null");
    this.isOpen = isOpen;
    this.actingParameters = actingParameters;
    this.parameters = enabled(parameters);
    this.requiresConfiguration = requiresConfiguration;
    this.requiresConnection = requiresConnection;
    this.partOrder = partOrder;
    this.providerName = providerName;
    this.providerId = enabled(providerId);
  }

  /**
   * @return the list of parameters that are required to execute the Value Provider resolution.
   * @deprecated since 1.4.0, use {@link ValueProviderModel#getParameters()} instead.
   */
  @Deprecated
  public List<String> getActingParameters() {
    return actingParameters;
  }

  /**
   * @return the list of parameters that the Value Provider takes into account for its resolution.
   */
  public Feature<List<ActingParameterModel>> getParameters() {
    if (parameters == null) {
      parameters = disabled();
    }
    return parameters;
  }

  /**
   * @return the order of this part in the value
   */
  public Integer getPartOrder() {
    return partOrder;
  }

  /**
   * @return The name of the element that provides values
   */
  public String getProviderName() {
    return providerName;
  }

  /**
   * @return a boolean indicating if the configuration is required to resolve the values
   */
  public boolean requiresConfiguration() {
    return requiresConfiguration;
  }


  /**
   * @return a boolean indicating if the connection is required to resolve the values
   */
  public boolean requiresConnection() {
    return requiresConnection;
  }

  /**
   * @return a boolean indicating if the calculated values should be considered as an open or closed set
   */
  public boolean isOpen() {
    return isOpen;
  }

  /**
   * @return The id of the element that provides values
   */
  public Feature<String> getProviderId() {
    if (providerId == null) {
      providerId = disabled();
    }
    return providerId;
  }

  @Override
  public boolean equals(Object o) {
    if (this == o) {
      return true;
    }

    if (o == null || getClass() != o.getClass()) {
      return false;
    }

    ValueProviderModel that = (ValueProviderModel) o;

    return new EqualsBuilder()
        .append(parameters, that.parameters)
        .append(partOrder, that.partOrder)
        .append(providerName, that.providerName)
        .append(requiresConnection, that.requiresConnection)
        .append(requiresConfiguration, that.requiresConfiguration)
        .append(isOpen, that.isOpen)
        .isEquals();
  }

  @Override
  public int hashCode() {
    return new HashCodeBuilder(17, 37)
        .append(parameters)
        .append(partOrder)
        .append(providerName)
        .append(requiresConnection)
        .append(requiresConfiguration)
        .append(isOpen)
        .toHashCode();
  }

  @Override
  public String toString() {
    return "ValueProviderModel{" +
        "providerName='" + providerName + '\'' +
        ", providerId=" + (providerId.isEnabled() ? providerId.get() : "DISABLED") +
        ", partOrder=" + partOrder +
        ", requiresConfiguration=" + requiresConfiguration +
        ", requiresConnection=" + requiresConnection +
        ", isOpen=" + isOpen +
        ", actingParameters=" + actingParameters +
        ", parameters=" + (parameters.isEnabled() ? parameters.get() : "DISABLED") +
        '}';
  }
}
