/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.tooling.client.api.feature;

import static org.apache.commons.lang3.builder.HashCodeBuilder.reflectionHashCode;

import org.mule.tooling.client.api.exception.FeatureNotSupportedException;

import java.util.function.Consumer;

/**
 * Represents a feature that could be enabled or not for a particular Mule Tooling API version working with a Mule version
 *
 * Note that the value (if enabled) could be null
 *
 * @param <T>: the feature that could be not-available
 * @since 1.1
 */
public class Feature<T> {

  private T value;
  private boolean enabled;

  private Feature(T value, boolean enabled) {
    this.value = value;
    this.enabled = enabled;
  }

  /**
   * @return a disabled feature, so it cannot be used
   */
  public static <T> Feature<T> disabled() {
    return new Feature(null, false);
  }

  /**
   * @param value to be wrapped by the feature
   * @return an enabled feature, with it's value
   */
  public static <T> Feature<T> enabled(T value) {
    return new Feature<>(value, true);
  }

  /**
   * @return the value, if the feature is enabled. Throws a FeatureNotSupportedException if it is disabled
   */
  public T get() {
    if (!isEnabled()) {
      throw new FeatureNotSupportedException("Feature is not supported for this Mule Tooling API version");
    }
    return this.value;
  }

  /**
   * @return whether the feature is enabled or not.
   */
  public boolean isEnabled() {
    return this.enabled;
  }

  /**
   * If the feature is enabled, apply the consumer to it's associated value
   * 
   * @param consumer to apply to the value, in case this feature is enabled
   */
  public void ifEnabled(Consumer<T> consumer) {
    if (isEnabled()) {
      consumer.accept(value);
    }
  }

  /**
   *
   * @param defaultValue in case the feature is enabled
   * @return the feature's value if it's enabled, the defaultValue otherwise.
   */
  public T orElse(T defaultValue) {
    return isEnabled() ? value : defaultValue;
  }

  @Override
  public int hashCode() {
    return reflectionHashCode(this);
  }

  @Override
  public boolean equals(Object o) {
    if (!(o instanceof Feature)) {
      return false;
    }
    Feature other = (Feature) o;
    if (!other.isEnabled() && !this.isEnabled()) {
      return true;
    }
    if (!other.isEnabled() || !this.isEnabled()) {
      return false;
    }

    if (this.value == null) {
      return other.value == null;
    } else {
      return this.value.equals(other.value);
    }
  }

  @Override
  public String toString() {
    return "Feature{" +
        "value=" + value +
        ", enabled=" + enabled +
        '}';
  }
}
