/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.tooling.client.api.value.resolver;

import static java.util.Collections.emptyList;
import static java.util.Collections.emptySet;
import static java.util.Collections.singletonList;
import static java.util.Collections.unmodifiableSet;

import java.util.List;
import java.util.Set;

/**
 * Result generated from resolving all {@link ResolvedValue}s allowed for a given parameter.
 *
 * @since 1.4.0
 */
public class ValueResolverResult {

  private Set<ResolvedValue> values;
  private List<ValueResolverFailure> resolvingFailures;

  public static ValueResolverResult success(Set<ResolvedValue> values) {
    return new ValueResolverResult(values, emptyList());
  }

  public static ValueResolverResult failure(List<ValueResolverFailure> failures) {
    return new ValueResolverResult(emptySet(), failures);
  }

  public static ValueResolverResult failure(ValueResolverFailure failure) {
    return failure(singletonList(failure));
  }

  private ValueResolverResult(Set<ResolvedValue> values,
                              List<ValueResolverFailure> resolvingFailures) {
    this.values = unmodifiableSet(values);
    this.resolvingFailures = resolvingFailures;
  }

  /**
   * @return The resolved {@link Set} of {@link ResolvedValue values}. In an error case this will return an empty {@link Set}.
   */
  public Set<ResolvedValue> getResolvedValues() {
    return values;
  }

  /**
   * @return A {@link List<ValueResolverFailure>} with {@link ValueResolverFailure}s failures with all the errors that occurred
   *         trying to resolve the {@link ResolvedValue values}
   */
  public List<ValueResolverFailure> getFailures() {
    return resolvingFailures;
  }

  /**
   * @return A boolean indicating if the resolution finished correctly or not.
   */
  public boolean isSuccess() {
    return resolvingFailures.isEmpty();
  }

}
