/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.tooling.client.internal.persistence;

import static org.mule.tooling.client.internal.persistence.ErrorModelToIdentifierSerializer.deserialize;
import static org.mule.tooling.client.internal.persistence.ErrorModelToIdentifierSerializer.serialize;

import org.mule.tooling.client.api.extension.model.ErrorModel;

import java.io.IOException;
import java.util.HashMap;
import java.util.Map;

import com.google.gson.JsonObject;
import com.google.gson.JsonParser;
import com.google.gson.TypeAdapter;
import com.google.gson.stream.JsonReader;
import com.google.gson.stream.JsonWriter;

/**
 * {@link TypeAdapter} implementation for {@link ErrorModel}, which serializes it as error identifiers. An error identifier
 * follows the {@link ComponentIdentifier} structure, {@code errorNamespace:errorType}.
 *
 * @since 1.4
 */
public class ErrorModelToIdentifierTypeAdapter extends TypeAdapter<ErrorModel> {

  public static final String TYPE = "type";
  public static final String NAMESPACE = "namespace";
  private Map<String, ErrorModel> errorModelRepository = new HashMap<>();

  public ErrorModelToIdentifierTypeAdapter(Map<String, ErrorModel> errorModelMap) {
    errorModelRepository = errorModelMap;
  }

  @Override
  public void write(JsonWriter out, ErrorModel value) throws IOException {
    out.value(serialize(value));
  }

  @Override
  public ErrorModel read(JsonReader in) throws IOException {
    String errorIdentifier;
    try {
      errorIdentifier = in.nextString();
    } catch (IllegalStateException e) {
      // backwards compatibility support to read verbose error from components instead of using only its component identifier
      JsonObject error = new JsonParser().parse(in).getAsJsonObject();
      String type = error.get(TYPE).getAsString();
      String namespace = error.get(NAMESPACE).getAsString();
      errorIdentifier = namespace + ":" + type;
    }
    return deserialize(errorIdentifier, errorModelRepository);
  }
}
