/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.tooling.client.internal.serialization;

import static org.mule.tooling.client.internal.util.Preconditions.checkState;

import com.esotericsoftware.kryo.ClassResolver;
import com.esotericsoftware.kryo.Kryo;
import com.esotericsoftware.kryo.Registration;
import com.esotericsoftware.kryo.Serializer;
import com.esotericsoftware.kryo.util.DefaultStreamFactory;
import com.esotericsoftware.kryo.util.MapReferenceResolver;

import java.util.HashMap;
import java.util.Map;

/**
 * Subclass of {@link Kryo} which uses consistent, repeteable and predictable classes ids. When Kryo serializes and object, it
 * doesn't store its canonical class name. Instead, it uses a sequential ID to refer to each encountered class and only stores
 * that id. That is fine when the types of all serialized objects is predictable. Because Tooling supports previous and next
 * versions of its implementation, this condition is not met.
 *
 * @since 1.2
 */
final class ExternalizableKryo extends Kryo {

  private Class<?> classBeingRegistered = null;
  private final Map<Class, Registration> class2Registration = new HashMap<>();

  private ClassSerializerFilter classSerializerFilter;
  private FieldSerializerFilter fieldSerializerFilter;

  public ExternalizableKryo(ClassSerializerFilter classSerializerFilter, FieldSerializerFilter fieldSerializerFilter,
                            ClassResolver classResolver) {
    super(classResolver, new MapReferenceResolver(), new DefaultStreamFactory());

    this.classSerializerFilter = classSerializerFilter;
    this.fieldSerializerFilter = fieldSerializerFilter;
  }

  public ExternalizableKryo(ClassResolver classResolver) {
    super(classResolver, new MapReferenceResolver(), new DefaultStreamFactory());
  }

  @Override
  public Registration register(Class type, Serializer serializer) {
    classBeingRegistered = type;
    try {
      return super.register(type, serializer);
    } finally {
      classBeingRegistered = null;
    }
  }

  @Override
  public int getNextRegistrationId() {
    checkState(classBeingRegistered != null, "Cannot generate an id while no class is being registered");
    return classBeingRegistered.getCanonicalName().hashCode();
  }

  protected Class<?> getClassBeingRegistered() {
    return classBeingRegistered;
  }

  @Override
  public Registration getRegistration(final Class type) {
    if (class2Registration.containsKey(type)) {
      return class2Registration.get(type);
    } else {
      Registration registration = null;

      for (Class searchType = type; registration == null && isRecurseRegistrationSearch(searchType); searchType =
          searchType.getSuperclass()) {
        registration = super.getRegistration(searchType);
        if (registration.getId() == -1) {
          registration = null;
        }
      }

      if (registration == null) {
        for (Class iFace : type.getInterfaces()) {
          registration = getRegistration(iFace);
          if (registration.getId() == -1) {
            registration = null;
          } else {
            break;
          }
        }
      }
      registration = registration != null ? registration : super.getRegistration(type);
      class2Registration.put(type, registration);
      return registration;
    }
  }

  private <T> boolean isRecurseRegistrationSearch(Class<T> searchType) {
    return searchType != null && !Enum.class.isAssignableFrom(searchType) && !Object.class.equals(searchType);
  }

  public ClassSerializerFilter getClassSerializerFilter() {
    return classSerializerFilter;
  }

  public FieldSerializerFilter getFieldSerializerFilter() {
    return this.fieldSerializerFilter;
  }
}
