/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */

package org.mule.tooling.client.internal.serialization;

import com.google.common.cache.Cache;
import com.google.common.cache.CacheBuilder;

import java.util.concurrent.ExecutionException;

/**
 * Implementation that uses the target {@link ClassLoader} to check for fields to be included or not in the serialization.
 */
public class ClassLoaderClassSerializerFilter implements ClassSerializerFilter {

  private ClassLoader targetClassLoader;
  private Cache<Class, Boolean> classesCache;

  /**
   * Creates an instance of the filter class loader serializer with the target class loader to check if classes can be
   * deserialized.
   *
   * @param targetClassLoader {@link ClassLoader} to check if the class should be included in serialization.
   */
  public ClassLoaderClassSerializerFilter(ClassLoader targetClassLoader) {
    this.targetClassLoader = targetClassLoader;
    classesCache = CacheBuilder.newBuilder()
        .build();
  }

  @Override
  public boolean shouldInclude(Class type) {
    try {
      return classesCache.get(type, () -> {
        try {
          targetClassLoader.loadClass(type.getName());
        } catch (ClassNotFoundException e) {
          return false;
        }
        return true;
      });
    } catch (ExecutionException e) {
      throw new RuntimeException(e.getCause());
    }
  }

}
