/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.tooling.client.api.extension.model.metadata;

import static java.util.Collections.emptyMap;
import static java.util.Collections.unmodifiableMap;
import static java.util.Optional.empty;
import static java.util.Optional.ofNullable;
import static org.apache.commons.lang3.StringUtils.isNotBlank;
import static org.mule.tooling.client.internal.util.Preconditions.checkArgument;

import java.util.Map;
import java.util.Objects;
import java.util.Optional;

/**
 * A definition for a of {@link org.mule.tooling.client.api.extension.model.ComponentModel} that contains all the information
 * regarding which dynamic {@link TypeResolver}s are used by this component.
 *
 * @since 1.4.0
 */
public class TypeResolversInformationModel {

  private final String category;
  private final ResolverInformation outputResolver;
  private final ResolverInformation attributesResolver;
  private final ResolverInformation keysResolver;
  private final boolean partialTypeKeyResolver;
  private final Map<String, ResolverInformation> inputResolvers;

  public TypeResolversInformationModel(String category,
                                       Map<String, ResolverInformation> inputResolvers,
                                       ResolverInformation outputResolver,
                                       ResolverInformation attributesResolver,
                                       ResolverInformation keysResolver,
                                       boolean partialTypeKeyResolver) {
    checkArgument(isNotBlank(category), "A Category name is required for a group of resolvers");
    this.category = category;

    this.outputResolver = outputResolver;
    this.attributesResolver = attributesResolver;
    this.keysResolver = keysResolver;
    this.partialTypeKeyResolver = partialTypeKeyResolver;

    this.inputResolvers = inputResolvers;
  }

  /**
   * Provides the name of the metadata category associated to the Component.
   *
   * @return category name
   */
  public String getCategoryName() {
    return category;
  }

  /**
   * Provides information of the output resolver (if any) associated to the Component.
   *
   * @return output resolver's {@link ResolverInformation}
   */
  public Optional<ResolverInformation> getOutputResolver() {
    return ofNullable(outputResolver);
  }

  /**
   * Provides information the output attributes resolver (if any) associated to the Component.
   *
   * @return output attributes resolver's {@link ResolverInformation}
   */
  public Optional<ResolverInformation> getAttributesResolver() {
    return ofNullable(attributesResolver);
  }

  /**
   * Provides information of the metadata keys resolver (if any) associated to the Component.
   *
   * @return keys resolver's {@link ResolverInformation}
   */
  public Optional<ResolverInformation> getKeysResolver() {
    return ofNullable(keysResolver);
  }

  /**
   * Provides information of the resolver (if any) associated to a given parameter.
   *
   * @param parameterName name of the parameter
   * @return {@link ResolverInformation} of the resolver associated to the parameter, if one exists.
   */
  public Optional<ResolverInformation> getParameterResolver(String parameterName) {
    return inputResolvers != null ? ofNullable(inputResolvers.get(parameterName)) : empty();
  }

  /**
   * Provides information of all the resolvers (if any) associated with input parameters
   *
   * @return {@link Map<String,ResolverInformation>} of the resolvers associated to input parameters.
   */
  public Map<String, ResolverInformation> getParameterResolvers() {
    return inputResolvers != null ? unmodifiableMap(inputResolvers) : emptyMap();
  }

  @Override
  public boolean equals(Object o) {
    if (this == o) {
      return true;
    }
    if (o == null || getClass() != o.getClass()) {
      return false;
    }
    TypeResolversInformationModel that = (TypeResolversInformationModel) o;
    return Objects.equals(category, that.category) &&
        Objects.equals(inputResolvers, that.inputResolvers) &&
        Objects.equals(outputResolver, that.outputResolver) &&
        Objects.equals(keysResolver, that.keysResolver) &&
        Objects.equals(attributesResolver, that.attributesResolver) &&
        Objects.equals(partialTypeKeyResolver, that.partialTypeKeyResolver);
  }

  @Override
  public int hashCode() {
    return Objects.hash(category, inputResolvers, outputResolver, keysResolver, attributesResolver, partialTypeKeyResolver);
  }

  @Override
  public String toString() {
    return "TypeResolversInformationModel{" +
        "category='" + category + '\'' +
        ", parameters=" + inputResolvers +
        ", outputResolver='" + outputResolver + '\'' +
        ", keysResolver='" + keysResolver + '\'' +
        ", attributesResolver='" + attributesResolver + '\'' +
        ", partialTypeKeyResolver='" + partialTypeKeyResolver + '\'' +
        '}';
  }
}
