/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.tooling.client.api.artifact.ast;

import static java.util.Collections.unmodifiableList;
import static java.util.Optional.empty;
import static java.util.Optional.ofNullable;
import static java.util.function.UnaryOperator.identity;
import static java.util.stream.Collectors.toMap;

import org.mule.metadata.api.model.MetadataType;
import org.mule.tooling.client.api.component.ComponentIdentifier;
import org.mule.tooling.client.api.component.ComponentType;
import org.mule.tooling.client.api.component.location.ComponentLocation;
import org.mule.tooling.client.api.component.location.SourceCodeLocation;
import org.mule.tooling.client.api.extension.model.ComponentModel;
import org.mule.tooling.client.api.extension.model.ExtensionModel;
import org.mule.tooling.client.api.extension.model.config.ConfigurationModel;
import org.mule.tooling.client.api.extension.model.connection.ConnectionProviderModel;

import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.stream.Stream;

/**
 * Represents the user configuration of a component (flow, config, message processor, etc) defined in an artifact.
 * <p>
 * Every {@code ComponentAst} represents the configuration of a core configuration or an extension configuration. Which
 * configuration element this object represents is identified by a {@link ComponentIdentifier} that can be retrieved using
 * {@code #getIdentifier()}.
 *
 * @since 1.4
 */
public class ComponentAst {

  private final String componentId;
  private final ComponentType componentType;
  private final ExtensionModel extension;
  private final ComponentGenerationInformation generationInformation;
  private final ComponentIdentifier identifier;
  private final ComponentLocation location;
  private final SourceCodeLocation metadata;
  private final Object model;
  private final MetadataType type;
  private final List<ComponentParameterAst> parameters;
  private final Map<String, ComponentParameterAst> parametersMap;
  private final List<ComponentAst> directChildren;

  public ComponentAst(String componentId,
                      ComponentType componentType,
                      ExtensionModel extension,
                      ComponentGenerationInformation generationInformation,
                      ComponentIdentifier identifier,
                      ComponentLocation location,
                      SourceCodeLocation metadata,
                      Object model,
                      MetadataType type,
                      List<ComponentParameterAst> parameters,
                      List<ComponentAst> directChildren) {
    this.componentId = componentId;
    this.componentType = componentType;
    this.extension = extension;
    this.generationInformation = generationInformation;
    this.identifier = identifier;
    this.location = location;
    this.metadata = metadata;
    this.model = model;
    this.type = type;
    this.parameters = unmodifiableList(parameters);
    this.parametersMap = parameters.stream()
        .filter(p -> p.getModel() != null)
        .collect(toMap(p -> p.getModel().getName(), identity()));
    this.directChildren = unmodifiableList(directChildren);
  }

  /**
   * @return the identifier for the configuration element this object represents.
   */
  public ComponentIdentifier getIdentifier() {
    return identifier;
  }

  /**
   * @return a general typification of the role of this component.
   */
  public ComponentType getComponentType() {
    return componentType;
  }

  /**
   * @return the location of the component in the configuration.
   */
  public ComponentLocation getLocation() {
    return location;
  }

  /**
   * @return the parser metadata for this component.
   */
  public SourceCodeLocation getMetadata() {
    return metadata;
  }

  /**
   * @return an object containing information about the generation of this component.
   */
  public ComponentGenerationInformation getGenerationInformation() {
    return generationInformation;
  }

  /**
   * @return the id of this component if it has one defined, or {@link Optional#empty() empty} otherwise.
   */
  public Optional<String> getComponentId() {
    return ofNullable(componentId);
  }

  /**
   * @return the extension that declares the model that represents this component.
   */
  public ExtensionModel getExtension() {
    return extension;
  }

  /**
   * A component may be represented by different kinds of models, depending on the actual type of this component. Possible values
   * may be, for instance, instances of {@link ConfigurationModel}, {@link ComponentModel} or {@link ConnectionProviderModel} (but
   * not necessarily limited to those).
   *
   * @param modelClass the class of the model this method should return.
   * @return the model that represents this component.
   */
  public <M> Optional<M> getModel(Class<M> modelClass) {
    if (modelClass.isInstance(model)) {
      return ofNullable((M) model);
    } else {
      return empty();
    }
  }

  /**
   *
   * @return the type that represents this component, if it is represented by a type.
   */
  public MetadataType getType() {
    return type;
  }

  // TODO MULE-19361 add this method
  // public ComponentParameterAst getParameter(String groupName, String paramName) {
  // // TODO Auto-generated method stub
  // return null;
  // }

  /**
   * @param paramName the name of the parameter to get AST for.
   * @return the AST of the parameter, or null if it doesn't exist.
   */
  public ComponentParameterAst getParameter(String paramName) {
    return getAllParameters().stream()
        .filter(p -> p.getModel() != null && p.getModel().getName().equals(paramName))
        .findAny()
        .orElse(null);
  }

  /**
   * A parameter may have a value either because it is explicitly set in the DSL, or because the extension model defines a default
   * value for it.
   *
   * @return the ASTs of all the parameters that have a value.
   */
  public List<ComponentParameterAst> getAllParameters() {
    return parameters;
  }

  /**
   * A parameter may have a value either because it is explicitly set in the DSL, or because the extension model defines a default
   * value for it.
   *
   * @return the ASTs of all the parameters that have a value with the name of the parameter as the key.
   */
  public Map<String, ComponentParameterAst> getParameters() {
    return parametersMap;
  }

  /**
   * @return a {@link Stream} for this component's children.
   */
  public List<ComponentAst> getDirectChildren() {
    return directChildren;
  }

}
