/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.tooling.client.api.component.location;

import static java.util.Optional.of;
import static java.util.Optional.ofNullable;

import org.mule.tooling.client.api.component.TypedComponentIdentifier;

import java.util.List;
import java.util.Objects;
import java.util.Optional;

/**
 * Provides information about the location of a component within an application.
 *
 * A component location describes where the component is defined in the configuration of the artifact.
 *
 * For instance:
 * <ul>
 * <li>COMPONENT_NAME - global component defined with name COMPONENT_NAME</li>
 * <li>FLOW_NAME/source - a source defined within a flow</li>
 * <li>FLOW_NAME/processors/0 - the first processor defined within a flow with name FLOW_NAME</li>
 * <li>FLOW_NAME/processors/4/1 - the first processors defined inside another processor which is positioned fifth within a flow
 * with name FLOW_NAME</li>
 * <li>FLOW_NAME/errorHandler/0 - the first on-error within the error handler</li>
 * <li>FLOW_NAME/0/errorHandler/3 - the third on-error within the error handler of the first element of the flow with name
 * FLOW_NAME</li>
 * </ul>
 *
 * @since 1.0
 */
public class ComponentLocation {

  private String location;
  // Previous version of Tooling Runtime Client will generate a ComponentLocation with this field instead of
  // SouceCodeLocation so in order to keep backward compatibility we leave these fields that will populated when
  // deserializing data from those versions.
  private String fileName;
  private Integer lineInFile;

  private SourceCodeLocation sourceCodeLocation;
  private TypedComponentIdentifier typedComponentIdentifier;
  private List<LocationPart> locationParts;

  // Just needed in order to serialize this object
  private ComponentLocation() {}

  public ComponentLocation(String location, TypedComponentIdentifier typedComponentIdentifier, List<LocationPart> locationParts,
                           Optional<SourceCodeLocation> sourceCodeLocation) {
    this.location = location;
    this.sourceCodeLocation = sourceCodeLocation.orElse(null);
    this.typedComponentIdentifier = typedComponentIdentifier;
    this.locationParts = locationParts;
  }

  /**
   * @return the unique absolute path of the component in the application.
   */
  public String getLocation() {
    return location;
  }

  /**
   * @return the config file of the application where this component is defined, if it was defined in a config file.
   */
  public Optional<String> getFileName() {
    if (fileName != null) {
      return of(fileName);
    }
    return getSourceCodeLocation().map(sourceCodeLocation -> sourceCodeLocation.getFileName());
  }

  /**
   * @return the line number in the config file of the application where this component is defined, if it was defined in a config
   *         file.
   */
  public Optional<Integer> getLineInFile() {
    if (lineInFile != null) {
      return of(lineInFile);
    }
    return getSourceCodeLocation().map(sourceCodeLocation -> sourceCodeLocation.getStartLine());
  }

  /**
   * @return the information of the location of the component in the configuration file. It may be empty if the components was
   *         added programmatically by the runtime.
   */
  public Optional<SourceCodeLocation> getSourceCodeLocation() {
    return ofNullable(sourceCodeLocation);
  }

  /**
   * @return the list of parts for the location. The location starts with the global element containing the component and
   *         continues with the next elements within the global element until the last part which is the component specific part.
   */
  public List<LocationPart> getParts() {
    return locationParts;
  }

  /**
   * @return the {@link TypedComponentIdentifier} of the component associated with this location
   */
  public TypedComponentIdentifier getComponentIdentifier() {
    return typedComponentIdentifier;
  }

  @Override
  public boolean equals(Object o) {
    if (this == o) {
      return true;
    }
    if (o == null || getClass() != o.getClass()) {
      return false;
    }
    ComponentLocation that = (ComponentLocation) o;
    return Objects.equals(getLocation(), that.getLocation()) &&
        Objects.equals(getSourceCodeLocation(), that.getSourceCodeLocation()) &&
        Objects.equals(typedComponentIdentifier, that.typedComponentIdentifier) &&
        Objects.equals(locationParts, that.locationParts);
  }

  @Override
  public int hashCode() {
    return Objects.hash(getLocation(), getSourceCodeLocation(), typedComponentIdentifier, locationParts);
  }

  @Override
  public String toString() {
    return this.getClass().getName() + "{" + location + " @ " + sourceCodeLocation.toString() + "}";
  }
}
