/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.tooling.client.api.component.location;

import static java.util.Optional.ofNullable;
import org.mule.tooling.client.api.component.TypedComponentIdentifier;

import java.util.Objects;
import java.util.Optional;

/**
 * A location part represent an specific location of a component within another component.
 *
 * @since 1.0
 */
public class LocationPart {

  private String path;
  private SourceCodeLocation sourceCodeLocation;
  private TypedComponentIdentifier identifier;

  private LocationPart() {}

  public LocationPart(String path, TypedComponentIdentifier identifier, Optional<SourceCodeLocation> sourceCodeLocation) {
    this.path = path;
    this.identifier = identifier;
    this.sourceCodeLocation = sourceCodeLocation.orElse(null);
  }

  /**
   * @return the string representation of the part
   */
  public String getPartPath() {
    return path;
  }

  /**
   * A location part represent an specific location of a component within another component.
   *
   * @since 4.0
   */
  public Optional<TypedComponentIdentifier> getPartIdentifier() {
    return ofNullable(identifier);
  }

  /**
   * @return the location of the component in the configuration files.
   */
  public Optional<SourceCodeLocation> getSourceCodeLocation() {
    return ofNullable(sourceCodeLocation);
  }


  @Override
  public boolean equals(Object o) {
    if (this == o) {
      return true;
    }
    if (o == null || getClass() != o.getClass()) {
      return false;
    }
    LocationPart that = (LocationPart) o;
    return Objects.equals(path, that.path) &&
        Objects.equals(getSourceCodeLocation(), that.getSourceCodeLocation()) &&
        Objects.equals(identifier, that.identifier);
  }

  @Override
  public int hashCode() {
    return Objects.hash(path, getSourceCodeLocation(), identifier);
  }
}
