/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.tooling.client.api.configuration.ssl;

import static java.util.Objects.requireNonNull;
import org.mule.tooling.client.internal.configuration.ssl.ImmutableKeyStoreConfig;

import java.io.File;

/**
 * Configuration for a key store.
 *
 * @since 1.0
 */
public interface KeyStoreConfig {

  /**
   * @return key store {@link File}.
   */
  File getKeyStoreFile();

  /**
   * @return the password for the key store.
   */
  String getKeyStorePassword();

  /**
   * @return a {@link Builder} to build a {@link KeyStoreConfig}.
   */
  static Builder builder() {
    return new Builder();
  }

  class Builder {

    private File keyStoreFile;
    private String keyStorePassword;

    /**
     * Sets the key store {@link File}.
     *
     * @param keyStoreFile key store {@link File}.
     * @return this
     */
    public Builder withKeyStoreFile(File keyStoreFile) {
      requireNonNull(keyStoreFile, "keyStoreFile cannot be null");
      this.keyStoreFile = keyStoreFile;
      return this;
    }

    /**
     * Sets the password for the key store.
     *
     * @param keyStorePassword password for the key store.
     * @return this
     */
    public Builder withKeyStorePassword(String keyStorePassword) {
      requireNonNull(keyStorePassword, "keyStorePassword cannot be null");
      this.keyStorePassword = keyStorePassword;
      return this;
    }

    /**
     * Builds the {@link KeyStoreConfig} object.
     *
     * @return {@link KeyStoreConfig} with the value sets.
     */
    public KeyStoreConfig build() {
      return new ImmutableKeyStoreConfig(this.keyStoreFile, this.keyStorePassword);
    }

  }

}
