/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.tooling.client.api.dataweave.validation;

import org.mule.metadata.api.model.MetadataType;
import org.mule.tooling.client.api.request.AbstractToolingRequest;

import java.util.Map;

/**
 * Request for {@link org.mule.tooling.client.api.dataweave.DataWeaveService#validate(DataWeaveValidationRequest)}. It provides
 * the context for DataWeave to validate an script, these properties are provided by DataSense propagation.
 *
 * @since 1.0
 */
public class DataWeaveValidationRequest extends AbstractToolingRequest {

  private String script;
  private Map<String, MetadataType> globalBindings;
  private Map<String, MetadataType> functionBindings;
  private MetadataType eventType;

  private DataWeaveValidationRequest(String script,
                                     Map<String, MetadataType> globalBindings,
                                     Map<String, MetadataType> functionBindings, MetadataType eventType) {
    this.script = script;
    this.globalBindings = globalBindings;
    this.functionBindings = functionBindings;
    this.eventType = eventType;
  }

  public String getScript() {
    return script;
  }

  public Map<String, MetadataType> getGlobalBindings() {
    return globalBindings;
  }

  public Map<String, MetadataType> getFunctionBindings() {
    return functionBindings;
  }

  public MetadataType getEventType() {
    return eventType;
  }

  /**
   * @return a new {@link Builder}.
   */
  public static DataWeaveValidationRequest.Builder builder() {
    return new Builder();
  }

  /**
   * Builder for {@link DataWeaveValidationRequest}.
   * 
   * @since 1.0
   */
  public static class Builder {

    private Map<String, MetadataType> globalBindings;
    private Map<String, MetadataType> functionBindings;
    private MetadataType eventType;
    private String script;

    private Builder() {
      // only accessible using the factory method
    }

    /**
     * Sets the globalBindings defined by the Mule Runtime to be used by DataWeave in order to validate the script. They should be
     * obtained from the DataSense propagation.
     *
     * @param globalBindings to be used by DataWeave in order to validate the script
     * @return this
     */
    public Builder withGlobalBindings(Map<String, MetadataType> globalBindings) {
      this.globalBindings = globalBindings;
      return this;
    }

    /**
     * Sets the functionBindings defined by the plugins declared by the application to be used by DataWeave in order to validate
     * the script. They should be obtained from the DataSense propagation.
     *
     * @param functionBindings provided by plugins to be used by DataWeave in order to validate the script
     * @return this
     */
    public Builder withFunctionBindings(Map<String, MetadataType> functionBindings) {
      this.functionBindings = functionBindings;
      return this;
    }

    /**
     * @param eventType {@link MetadataType} of the Mule Event at the point were the script is going to be evaluated.
     * @return this
     */
    public Builder withEventType(MetadataType eventType) {
      this.eventType = eventType;
      return this;
    }

    /**
     * @param script to be validated
     * @return this
     */
    public Builder withScript(String script) {
      this.script = script;
      return this;
    }

    /**
     * @return a {@link DataWeaveValidationRequest}
     */
    public DataWeaveValidationRequest build() {
      return new DataWeaveValidationRequest(script, globalBindings, functionBindings, eventType);
    }
  }
}
