/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.tooling.client.api.extension.model.construct;


import static java.util.Collections.emptySet;
import static java.util.Collections.unmodifiableSet;
import static java.util.Optional.ofNullable;
import static org.mule.tooling.client.api.feature.Feature.disabled;
import static org.mule.tooling.client.api.feature.Feature.enabled;

import org.mule.tooling.client.api.extension.model.ComponentModel;
import org.mule.tooling.client.api.extension.model.DisplayModel;
import org.mule.tooling.client.api.extension.model.ErrorModel;
import org.mule.tooling.client.api.extension.model.StereotypeModel;
import org.mule.tooling.client.api.extension.model.connection.ConnectionProviderModel;
import org.mule.tooling.client.api.extension.model.deprecated.DeprecationModel;
import org.mule.tooling.client.api.extension.model.metadata.TypeResolversInformationModel;
import org.mule.tooling.client.api.extension.model.nested.NestableElementModel;
import org.mule.tooling.client.api.extension.model.parameter.ParameterGroupModel;
import org.mule.tooling.client.api.feature.Feature;

import java.util.HashSet;
import java.util.List;
import java.util.Optional;
import java.util.Set;

import org.apache.commons.lang3.builder.ToStringBuilder;

/**
 * DTO for a ConstructModel.
 *
 * @since 1.0
 */
public class ConstructModel implements ComponentModel {

  private String name;
  private String description;
  private List<ParameterGroupModel> parameterGroupModels;
  private DisplayModel displayModel;
  private StereotypeModel stereotype;
  private List<? extends NestableElementModel> nestedComponents;
  private boolean allowsTopLevelDefinition;
  private Set<ErrorModel> errorModels;
  private Feature<DeprecationModel> deprecationModel;
  private Feature<TypeResolversInformationModel> typeResolversInformationModel;

  // There is no point in making this a feature. If disabled, we can just return an empty set
  private Set<String> semanticTerms;

  @Deprecated
  public ConstructModel(String name,
                        String description,
                        List<ParameterGroupModel> parameterGroupModels,
                        DisplayModel displayModel,
                        StereotypeModel stereotype,
                        List<? extends NestableElementModel> nestedComponents,
                        boolean allowsTopLevelDefinition,
                        Set<ErrorModel> errorModels,
                        DeprecationModel deprecationModel) {
    this(name, description, parameterGroupModels, displayModel, stereotype, nestedComponents, allowsTopLevelDefinition,
         errorModels, deprecationModel, null);
  }

  /**
   * Creates a new instance with the given state
   *
   * @param name                          the operation's name. Cannot be blank
   * @param description                   the operation's descriptor
   * @param parameterGroupModels          a {@link List} with the operation's {@link ParameterGroupModel parameter group models}
   * @param displayModel                  a model which contains directive about how this operation is displayed in the UI
   * @param stereotype                    the construct {@link StereotypeModel}
   * @param nestedComponents              a {@link List} with the components contained by this model
   * @param allowsTopLevelDefinition      whether or not {@code this} model can be declared as a root component in the application
   * @param errorModels                   {@link Set} of {@link ErrorModel} error models
   * @param deprecationModel              a {@link DeprecationModel} describing if the construct is deprecated. A null value means
   *                                      it is not deprecated.
   * @param typeResolversInformationModel {@link TypeResolversInformationModel} with information about metadata resolvers for this
   *                                      component.
   * @throws IllegalArgumentException if {@code name} is blank or {@code executorFactory} is {@code null}
   */
  public ConstructModel(String name,
                        String description,
                        List<ParameterGroupModel> parameterGroupModels,
                        DisplayModel displayModel,
                        StereotypeModel stereotype,
                        List<? extends NestableElementModel> nestedComponents,
                        boolean allowsTopLevelDefinition,
                        Set<ErrorModel> errorModels,
                        DeprecationModel deprecationModel,
                        TypeResolversInformationModel typeResolversInformationModel) {
    this(name, description, parameterGroupModels, displayModel, stereotype, nestedComponents, allowsTopLevelDefinition,
         errorModels, deprecationModel, typeResolversInformationModel, emptySet());
  }

  /**
   * Creates a new instance with the given state
   *
   * @param name                          the operation's name. Cannot be blank
   * @param description                   the operation's descriptor
   * @param parameterGroupModels          a {@link List} with the operation's {@link ParameterGroupModel parameter group models}
   * @param displayModel                  a model which contains directive about how this operation is displayed in the UI
   * @param stereotype                    the construct {@link StereotypeModel}
   * @param nestedComponents              a {@link List} with the components contained by this model
   * @param allowsTopLevelDefinition      whether or not {@code this} model can be declared as a root component in the application
   * @param errorModels                   {@link Set} of {@link ErrorModel} error models
   * @param deprecationModel              a {@link DeprecationModel} describing if the construct is deprecated. A null value means
   *                                      it is not deprecated.
   * @param typeResolversInformationModel {@link TypeResolversInformationModel} with information about metadata resolvers for this
   *                                      component.
   * @param semanticTerms                 a {@link Set} of semantic terms which describe the component's meaning and effect
   * @throws IllegalArgumentException if {@code name} is blank or {@code executorFactory} is {@code null}
   * @since 1.4.0
   */
  public ConstructModel(String name,
                        String description,
                        List<ParameterGroupModel> parameterGroupModels,
                        DisplayModel displayModel,
                        StereotypeModel stereotype,
                        List<? extends NestableElementModel> nestedComponents,
                        boolean allowsTopLevelDefinition,
                        Set<ErrorModel> errorModels,
                        DeprecationModel deprecationModel,
                        TypeResolversInformationModel typeResolversInformationModel,
                        Set<String> semanticTerms) {
    this.name = name;
    this.description = description;
    this.parameterGroupModels = parameterGroupModels;
    this.displayModel = displayModel;
    this.stereotype = stereotype;
    this.nestedComponents = nestedComponents;
    this.allowsTopLevelDefinition = allowsTopLevelDefinition;
    this.errorModels = errorModels;
    this.deprecationModel = enabled(deprecationModel);
    this.typeResolversInformationModel = enabled(typeResolversInformationModel);
    this.semanticTerms = new HashSet<>(semanticTerms);
  }

  public boolean allowsTopLevelDeclaration() {
    return allowsTopLevelDefinition;
  }

  public String getName() {
    return name;
  }

  public String getDescription() {
    return description;
  }

  public List<ParameterGroupModel> getParameterGroupModels() {
    return parameterGroupModels;
  }

  public Optional<DisplayModel> getDisplayModel() {
    return ofNullable(displayModel);
  }

  public StereotypeModel getStereotype() {
    return stereotype;
  }

  public List<? extends NestableElementModel> getNestedComponents() {
    return nestedComponents;
  }

  public boolean isAllowsTopLevelDefinition() {
    return allowsTopLevelDefinition;
  }

  public Set<ErrorModel> getErrorModels() {
    return errorModels;
  }

  public Feature<TypeResolversInformationModel> getTypeResolversInformationModel() {
    if (typeResolversInformationModel == null) {
      typeResolversInformationModel = disabled();
    }
    return typeResolversInformationModel;
  }

  public Feature<DeprecationModel> getDeprecationModel() {
    if (deprecationModel == null) {
      deprecationModel = disabled();
    }
    return deprecationModel;
  }

  @Override
  public Set<String> getSemanticTerms() {
    if (semanticTerms == null) {
      this.semanticTerms = emptySet();
    }
    return unmodifiableSet(semanticTerms);
  }

  // Helper methods

  public Optional<ParameterGroupModel> getParameterGroupModel(String name) {
    return parameterGroupModels.stream().filter(model -> model.getName().equals(name)).findFirst();
  }

  @Override
  public int hashCode() {
    return this.name.hashCode();
  }

  @Override
  public boolean equals(Object obj) {
    return this.getClass().isInstance(obj) && this.name.equals(((ConnectionProviderModel) obj).getName());
  }

  @Override
  public String toString() {
    return ToStringBuilder.reflectionToString(this);
  }

}
