/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.tooling.client.api.extension.model.parameter;

import static java.util.Optional.ofNullable;
import org.mule.tooling.client.api.extension.model.DisplayModel;
import org.mule.tooling.client.api.extension.model.LayoutModel;

import java.util.List;
import java.util.Optional;

import org.apache.commons.lang3.builder.ToStringBuilder;

/**
 * A group is a logical way to display one or more parameters together.
 *
 * @since 1.0
 */
public class ParameterGroupModel {

  public static String DEFAULT_GROUP_NAME = "General";
  public static String ADVANCED = "Advanced";
  public static String OUTPUT = "Output";
  public static String CONNECTION = "Connection";

  private String name;
  private String description;
  private DisplayModel displayModel;
  private LayoutModel layoutModel;
  private List<ParameterModel> parameterModels;
  private List<ExclusiveParametersModel> exclusiveParametersModels;
  private boolean showInDsl;

  // Just needed in order to serialize this object
  private ParameterGroupModel() {}

  public ParameterGroupModel(String name, String description,
                             DisplayModel displayModel, LayoutModel layoutModel,
                             List<ParameterModel> parameterModels,
                             List<ExclusiveParametersModel> exclusiveParametersModels, boolean showInDsl) {
    this.name = name;
    this.description = description;
    this.displayModel = displayModel;
    this.layoutModel = layoutModel;
    this.parameterModels = parameterModels;
    this.exclusiveParametersModels = exclusiveParametersModels;
    this.showInDsl = showInDsl;
  }

  public String getName() {
    return this.name;
  }

  public String getDescription() {
    return description;
  }

  public Optional<DisplayModel> getDisplayModel() {
    return ofNullable(displayModel);
  }

  public Optional<LayoutModel> getLayoutModel() {
    return ofNullable(layoutModel);
  }

  public List<ParameterModel> getParameterModels() {
    return parameterModels;
  }

  public List<ExclusiveParametersModel> getExclusiveParametersModels() {
    return exclusiveParametersModels;
  }

  public boolean isShowInDsl() {
    return showInDsl;
  }

  // Helper methods

  public Optional<ParameterModel> getParameterModel(String name) {
    return parameterModels.stream().filter(model -> model.getName().equals(name)).findFirst();
  }

  @Override
  public int hashCode() {
    return this.name.hashCode();
  }

  @Override
  public boolean equals(Object obj) {
    return this.getClass().isInstance(obj) && this.name.equals(((ParameterGroupModel) obj).getName());
  }

  @Override
  public String toString() {
    return ToStringBuilder.reflectionToString(this);
  }

}
