/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.tooling.client.api.metadata;

import java.util.List;

import static java.util.Arrays.asList;
import static java.util.Collections.emptyList;

/**
 * Container for the Metadata fetch operations results. Allows to communicate errors without propagating exceptions to the
 * Metadata fetching service
 *
 * @param <T> return type of the Metadata resolving operation.
 * @since 1.0
 */
public final class MetadataResult<T> {

  T value;
  List<MetadataFailure> failures;

  private MetadataResult(T value, List<MetadataFailure> failures) {
    this.value = value;
    this.failures = failures;
  }

  public static <T> MetadataResult<T> success(T payload) {
    return new MetadataResult<T>(payload, emptyList());
  }

  public static <T> MetadataResult<T> failure(MetadataFailure... failures) {
    return new MetadataResult<>(null, asList(failures));
  }

  public static <T> MetadataResult<T> failure(List<MetadataFailure> failures) {
    return new MetadataResult<>(null, failures);
  }


  /**
   * @return the object returned by the invoked Metadata operation
   */
  public T get() {
    return value;
  }

  /**
   * @return true if the operation was completed without errors, false otherwise
   */
  public boolean isSuccess() {
    return failures == null || failures.isEmpty();
  }

  /**
   * If {@link this#isSuccess} is false, then a {@link MetadataFailure} instance is provided in order to describe the error that
   * occurred during the invocation.
   *
   * @return a {@link List} of {@link MetadataFailure}s describing the errors that occurred during the invocation if at least one
   *         occurred.
   */
  public List<MetadataFailure> getFailures() {
    return failures;
  }


}
