/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.tooling.client.api.value;

import static java.util.Collections.unmodifiableSet;
import static java.util.Optional.ofNullable;

import java.util.Optional;
import java.util.Set;

/**
 * Represents the result of the resolution of the possible values of a element capable of resolve {@link Value values}.
 *
 * @since 1.0
 */
public class ValueResult {

  private Set<Value> values;
  private ResolvingFailure resolvingFailure;

  public ValueResult(Set<Value> values, ResolvingFailure resolvingFailure) {
    this.values = unmodifiableSet(values);
    this.resolvingFailure = resolvingFailure;
  }

  /**
   * @return The resolved {@link Set} of {@link Value values}. In an error case this will return an empty {@link Set}.
   */
  public Set<Value> getValues() {
    return values;
  }

  /**
   * @return An {@link Optional} {@link ResolvingFailure failure} with the error that occurred trying to resolve the {@link Value
   *         values}
   */
  public Optional<ResolvingFailure> getFailure() {
    return ofNullable(resolvingFailure);
  }

  /**
   * @return A boolean indicating if the resolution finished correctly or not.
   */
  public boolean isSuccess() {
    return !getFailure().isPresent();
  }
}
