/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.tooling.client.internal.persistence;

import static java.util.Optional.empty;
import org.mule.metadata.api.annotation.TypeAnnotation;
import org.mule.metadata.persistence.api.TypeAnnotationSerializerExtenderFallback;

import java.util.Optional;

/**
 * {@link TypeAnnotationSerializerExtenderFallback} implementation that allows Tooling to avoid trying to serialize or deserialize
 * a {@link org.mule.metadata.api.annotation.TypeAnnotation} that is not handled by this API.
 *
 * @since 1.0
 */
public class IgnoreMissingTypeAnnotationFallback implements TypeAnnotationSerializerExtenderFallback {

  @Override
  public Optional<Class<? extends TypeAnnotation>> fallbackFor(String s) {
    try {
      return Optional.of(s)
          .filter(className -> className.startsWith("org.mule.metadata."))
          .flatMap(this::resolveClass);
    } catch (Throwable e) {
      return Optional.empty();
    }
  }

  private Optional<Class<? extends TypeAnnotation>> resolveClass(String className) {
    try {
      Class<? extends TypeAnnotation> annotationClass = (Class<? extends TypeAnnotation>) Class.forName(className);
      return Optional.of(annotationClass);
    } catch (ClassNotFoundException e) {
      return Optional.empty();
    }
  }
}
