/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.tooling.client.internal.persistence;

import static java.lang.String.format;
import org.mule.tooling.client.api.extension.model.nested.NestableElementModel;
import org.mule.tooling.client.api.extension.model.nested.NestableElementModelVisitor;
import org.mule.tooling.client.api.extension.model.nested.NestedChainModel;
import org.mule.tooling.client.api.extension.model.nested.NestedComponentModel;
import org.mule.tooling.client.api.extension.model.nested.NestedRouteModel;

import com.google.gson.Gson;
import com.google.gson.TypeAdapter;
import com.google.gson.TypeAdapterFactory;
import com.google.gson.reflect.TypeToken;

/**
 * {@link TypeAdapterFactory} for {@link NestableElementModel}
 */
public class NestedElementModelTypeAdapter extends KindEnrichedTypeAdapter<NestableElementModel> {

  private static final String COMPONENT_KIND = "component";
  private static final String CHAIN_KIND = "chain";
  private static final String ROUTE_KIND = "route";

  public NestedElementModelTypeAdapter(TypeAdapterFactory typeAdapterFactory, Gson gson) {
    super(typeAdapterFactory, gson);
  }

  @Override
  protected String getKind(NestableElementModel value) {
    Reference<String> kind = new Reference<>();

    value.accept(new NestableElementModelVisitor() {

      @Override
      public void visit(NestedComponentModel component) {
        kind.set(COMPONENT_KIND);
      }

      @Override
      public void visit(NestedChainModel component) {
        kind.set(CHAIN_KIND);
      }

      @Override
      public void visit(NestedRouteModel component) {
        kind.set(ROUTE_KIND);
      }
    });

    return kind.get();
  }

  @Override
  protected TypeAdapter<NestableElementModel> getDelegateAdapter(String kind) {
    Class clazz;
    switch (kind) {
      case ROUTE_KIND:
        clazz = NestedRouteModel.class;
        break;
      case CHAIN_KIND:
        clazz = NestedChainModel.class;
        break;
      case COMPONENT_KIND:
        clazz = NestedComponentModel.class;
        break;
      default:
        throw new IllegalArgumentException(format("Unknown kind [%s] for a NestedElementModel", kind));
    }

    return gson.getDelegateAdapter(typeAdapterFactory, TypeToken.get(clazz));
  }
}
