/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */

package org.mule.tooling.client.bootstrap.internal.wrapper;

import static java.util.Objects.requireNonNull;
import org.mule.tooling.client.api.exception.NoSuchApplicationException;
import org.mule.tooling.client.api.message.history.MessageHistory;
import org.mule.tooling.client.api.message.history.MessageHistoryService;
import org.mule.tooling.client.bootstrap.internal.reflection.Dispatcher;
import org.mule.tooling.client.internal.serialization.XStreamClientSerializer;

import com.google.common.collect.ImmutableList;

/**
 * Client side implementation for {@link MessageHistoryService} that uses reflection and works as a bridge
 * between {@link ClassLoader classLoaders}.
 *
 * @since 1.0
 */
public class MessageHistoryServiceWrapper implements MessageHistoryService {

  private Dispatcher dispatcher;
  private XStreamClientSerializer serializer;

  public MessageHistoryServiceWrapper(Dispatcher dispatcher,
                                      XStreamClientSerializer serializer) {
    requireNonNull(dispatcher, "dispatcher cannot be null");
    requireNonNull(serializer, "serializer cannot be null");

    this.dispatcher = dispatcher;
    this.serializer = serializer;
  }

  @Override
  public void enable(String applicationName) throws NoSuchApplicationException {
    dispatcher.dispatchRemoteMethod("enable", ImmutableList.of(String.class),
                                    ImmutableList.of(serializer.serialize(applicationName)));
  }

  @Override
  public MessageHistory consume(String applicationName, int chunkSize) {
    String result = (String) dispatcher.dispatchRemoteMethod("consume",
                                                             ImmutableList.of(String.class, Integer.class),
                                                             ImmutableList.of(serializer.serialize(applicationName),
                                                                              serializer.serialize(chunkSize)));
    return serializer.deserialize(result);
  }

  @Override
  public void disable(String applicationName) {
    dispatcher.dispatchRemoteMethod("disable",
                                    ImmutableList.of(String.class),
                                    ImmutableList.of(serializer.serialize(applicationName)));
  }

}
