/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */

package org.mule.tooling.client.bootstrap.internal.wrapper;

import static java.lang.String.format;
import static java.lang.Thread.currentThread;
import static java.util.Objects.requireNonNull;
import static java.util.ServiceLoader.load;
import org.mule.maven.client.api.model.MavenConfiguration;
import org.mule.tooling.client.api.Disposable;
import org.mule.tooling.client.api.ToolingRuntimeClient;
import org.mule.tooling.client.api.ToolingRuntimeClientBuilderFactory;
import org.mule.tooling.client.bootstrap.internal.reflection.Dispatcher;

import java.util.ServiceLoader;
import java.util.concurrent.ExecutorService;

/**
 * Client side implementation for {@link ToolingRuntimeClientBuilderFactory} that uses reflection and works as a bridge
 * between {@link ClassLoader classLoaders}.
 *
 * @since 1.0
 */
public class ToolingRuntimeClientBuilderFactoryWrapper implements ToolingRuntimeClientBuilderFactory, Disposable {

  private ClassLoader toolingClassLoader;
  private Dispatcher dispatcher;
  private MavenConfiguration mavenConfiguration;

  public ToolingRuntimeClientBuilderFactoryWrapper(ClassLoader toolingClassLoader,
                                                   ExecutorService executorService, MavenConfiguration mavenConfiguration) {
    requireNonNull(toolingClassLoader, "toolingClassLoader cannot be null");
    requireNonNull(executorService, "executorService cannot be null");
    requireNonNull(mavenConfiguration, "mavenConfiguration cannot be null");

    this.toolingClassLoader = toolingClassLoader;
    this.dispatcher = new Dispatcher(discoverToolingRuntimeClientFactory(), toolingClassLoader, executorService);
    this.mavenConfiguration = mavenConfiguration;
  }

  private Object discoverToolingRuntimeClientFactory() {
    final ClassLoader contextClassLoader = currentThread().getContextClassLoader();
    try {
      currentThread().setContextClassLoader(toolingClassLoader);
      ServiceLoader<? extends Object> serviceLoader =
          load(toolingClassLoader.loadClass(ToolingRuntimeClientBuilderFactory.class.getName()), toolingClassLoader);
      if (!serviceLoader.iterator().hasNext()) {
        throw new IllegalStateException("No service found for: '" + ToolingRuntimeClientBuilderFactory.class.getName()
            + "'");
      }

      return serviceLoader.iterator().next();
    } catch (ClassNotFoundException e) {
      throw new IllegalStateException(format(
                                             "ClassLoader for Tooling was not correctly created as there is no implementation of %s",
                                             ToolingRuntimeClientBuilderFactory.class.getName()),
                                      e);
    } finally {
      currentThread().setContextClassLoader(contextClassLoader);
    }
  }

  @Override
  public void dispose() {
    dispatcher.dispatchRemoteMethod("dispose");
  }

  @Override
  public ToolingRuntimeClient.Builder create() {
    Object builderTarget = dispatcher.dispatchRemoteMethod("create");
    ToolingRuntimeClient.Builder builder = new BuilderWrapper(toolingClassLoader, dispatcher.newReflectionInvoker(builderTarget));
    // By default it mavenConfiguration from bootstrap is inherited
    builder.withMavenConfiguration(mavenConfiguration);
    return builder;
  }

}
