/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.tooling.client.bootstrap.internal;

import org.mule.tooling.client.api.exception.ToolingException;

import java.io.IOException;
import java.io.InputStream;
import java.util.HashMap;
import java.util.Map;
import java.util.Properties;

import static java.lang.System.getProperty;

/**
 * Resolves the Mule Tooling API implementation version that corresponds to a specific version of the Mule Runtime
 * Priority order:
 *    - System Property, if set.
 *    - Property File, if exists.
 *    - Otherwise: Same as the runtime version
 *    
 * @since 1.0
 */
public class VersionResolver {

  public static final String TOOLING_VERSION_MAPPING_FILE = "mule-tooling-version.properties";
  public static final String TOOLING_VERSION_PROPERTY = "mule.tolling.version";
  public static final String TOOLING_VERSION_PROPERTY_SEPARATOR = ",";
  public static final String TOOLING_VERSION_PROPERTY_MAPPING_SEPARATOR = ":";

  private static Map<String, String> getToolingMappings(String property) {
    Map<String, String> map = new HashMap<>();
    if (property == null) {
      return map;
    }

    for (String version : property.split(TOOLING_VERSION_PROPERTY_SEPARATOR)) {
      String sep[] = version.split(TOOLING_VERSION_PROPERTY_MAPPING_SEPARATOR);
      String muleVersion = sep[0];
      String mapedToolingVersion = sep[1];
      map.put(muleVersion, mapedToolingVersion);
    }

    return map;
  }

  public static String mapToolingVersion(ClassLoader classLoader, String runtimeVersion) {
    Map<String, String> toolingMappings = getToolingMappings(getProperty(TOOLING_VERSION_PROPERTY));
    if (toolingMappings.containsKey(runtimeVersion)) {
      return toolingMappings.get(runtimeVersion);
    }

    Properties p = new Properties();
    try (InputStream is = classLoader.getResourceAsStream(TOOLING_VERSION_MAPPING_FILE)) {
      if (is == null) {
        return runtimeVersion;
      }

      p.load(is);

      String result = (String) p.get(runtimeVersion);
      if (result != null) {
        return result;
      } else {
        return runtimeVersion;
      }

    } catch (IOException e) {
      throw new ToolingException("Error while reading file " + TOOLING_VERSION_MAPPING_FILE, e);
    }
  }

}
