/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */

package org.mule.tooling.client.bootstrap.internal.wrapper;

import static java.util.Objects.requireNonNull;
import org.mule.tooling.client.api.dataweave.DataWeaveModuleOption;
import org.mule.tooling.client.api.dataweave.DataWeavePreviewRequest;
import org.mule.tooling.client.api.dataweave.DataWeavePreviewResponse;
import org.mule.tooling.client.api.dataweave.DataWeaveService;
import org.mule.tooling.client.api.dataweave.validation.DataWeaveValidationRequest;
import org.mule.tooling.client.api.el.ValidationResult;
import org.mule.tooling.client.bootstrap.internal.reflection.Dispatcher;
import org.mule.tooling.client.internal.serialization.XStreamClientSerializer;

import com.google.common.collect.ImmutableList;

import java.util.List;

/**
 * Client side implementation for {@link DataWeaveService} that uses reflection and works as a bridge
 * between {@link ClassLoader classLoaders}.
 *
 * @since 1.0
 */
public class DataWeaveServiceWrapper implements DataWeaveService {

  private Dispatcher dispatcher;
  private XStreamClientSerializer serializer;

  public DataWeaveServiceWrapper(Dispatcher dispatcher,
                                 XStreamClientSerializer serializer) {
    requireNonNull(dispatcher, "dispatcher cannot be null");
    requireNonNull(serializer, "serializer cannot be null");

    this.dispatcher = dispatcher;
    this.serializer = serializer;
  }

  @Override
  public DataWeavePreviewResponse execute(DataWeavePreviewRequest parameters) {
    String result = (String) dispatcher.dispatchRemoteMethod("execute",
                                                             ImmutableList.of(DataWeavePreviewRequest.class),
                                                             ImmutableList.of(serializer.serialize(parameters)));
    return serializer.deserialize(result);
  }

  @Override
  public ValidationResult validate(DataWeaveValidationRequest request) {
    String result = (String) dispatcher.dispatchRemoteMethod("validate",
                                                             ImmutableList.of(DataWeaveValidationRequest.class),
                                                             ImmutableList.of(serializer.serialize(request)));
    return serializer.deserialize(result);
  }

  @Override
  public List<DataWeaveModuleOption> getReaderOptions(String contentType) {
    String result = (String) dispatcher.dispatchRemoteMethod("getReaderOptions",
                                                             ImmutableList.of(String.class),
                                                             ImmutableList.of(serializer.serialize(contentType)));
    return serializer.deserialize(result);
  }

  @Override
  public List<DataWeaveModuleOption> getWriterOptions(String contentType) {
    String result = (String) dispatcher.dispatchRemoteMethod("getWriterOptions",
                                                             ImmutableList.of(String.class),
                                                             ImmutableList.of(serializer.serialize(contentType)));
    return serializer.deserialize(result);
  }

}
