/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */

package org.mule.tooling.client.bootstrap.internal.wrapper;

import static java.util.Objects.requireNonNull;
import static org.mule.tooling.client.internal.serialization.SerializationUtils.deserialize;
import static org.mule.tooling.client.internal.serialization.SerializationUtils.serialize;
import org.mule.runtime.app.declaration.api.ArtifactDeclaration;
import org.mule.tooling.client.api.artifact.declaration.ArtifactSerializationException;
import org.mule.tooling.client.api.artifact.declaration.ArtifactSerializationService;
import org.mule.tooling.client.api.artifact.declaration.request.JsonArtifactDeserializationRequest;
import org.mule.tooling.client.api.artifact.declaration.request.JsonArtifactSerializationRequest;
import org.mule.tooling.client.api.artifact.declaration.request.XmlArtifactDeserializationRequest;
import org.mule.tooling.client.api.artifact.declaration.request.XmlArtifactSerializationRequest;
import org.mule.tooling.client.bootstrap.internal.reflection.Dispatcher;

import com.google.common.collect.ImmutableList;

/**
 * Client side implementation for {@link ArtifactSerializationService} that uses reflection and works as a bridge
 * between {@link ClassLoader classLoaders}.
 *
 * @since 1.0
 */
public class ArtifactSerializationServiceWrapper implements ArtifactSerializationService {

  private Dispatcher dispatcher;

  public ArtifactSerializationServiceWrapper(Dispatcher dispatcher) {
    requireNonNull(dispatcher, "dispatcher cannot be null");

    this.dispatcher = dispatcher;
  }

  @Override
  public String toXml(XmlArtifactSerializationRequest request) throws ArtifactSerializationException {
    String result = (String) dispatcher.dispatchRemoteMethod("toXml",
                                                             -1,
                                                             ImmutableList.of(XmlArtifactSerializationRequest.class),
                                                             ImmutableList.of(serialize(request)));
    return deserialize(result);
  }

  @Override
  public ArtifactDeclaration fromXml(XmlArtifactDeserializationRequest request) throws ArtifactSerializationException {
    return deserialize((String) dispatcher.dispatchRemoteMethod("fromXml",
                                                                -1,
                                                                ImmutableList.of(XmlArtifactDeserializationRequest.class),
                                                                ImmutableList.of(serialize(request))));
  }

  @Override
  public String toJson(JsonArtifactSerializationRequest request) throws ArtifactSerializationException {
    String result = (String) dispatcher.dispatchRemoteMethod("toJson",
                                                             -1,
                                                             ImmutableList.of(JsonArtifactSerializationRequest.class),
                                                             ImmutableList.of(serialize(request)));
    return deserialize(result);
  }

  @Override
  public ArtifactDeclaration fromJson(JsonArtifactDeserializationRequest request)
      throws ArtifactSerializationException {
    return deserialize((String) dispatcher.dispatchRemoteMethod("fromJson",
                                                                -1,
                                                                ImmutableList.of(JsonArtifactDeserializationRequest.class),
                                                                ImmutableList.of(serialize(request))));
  }

}
