/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */

package org.mule.tooling.client.bootstrap.internal.wrapper;

import static java.util.Objects.requireNonNull;
import static org.mule.tooling.client.internal.serialization.SerializationUtils.serialize;

import org.mule.tooling.client.api.ToolingRuntimeClient;
import org.mule.tooling.client.api.artifact.ToolingArtifact;
import org.mule.tooling.client.api.artifact.declaration.ArtifactSerializationService;
import org.mule.tooling.client.api.artifact.dsl.DslSyntaxResolverService;
import org.mule.tooling.client.api.exception.ToolingArtifactNotFoundException;
import org.mule.tooling.client.api.extension.ExtensionModelService;
import org.mule.tooling.client.api.icons.ExtensionIconsService;
import org.mule.tooling.client.api.message.history.MessageHistoryService;
import org.mule.tooling.client.bootstrap.internal.reflection.Dispatcher;

import java.net.URL;
import java.util.Map;

import com.google.common.collect.ImmutableList;

/**
 * Client side implementation for {@link ToolingRuntimeClient} that uses reflection and works as a bridge
 * between {@link ClassLoader classLoaders}.
 *
 * @since 1.0
 */
public class ToolingRuntimeClientWrappper implements ToolingRuntimeClient {

  private Dispatcher dispatcher;

  public ToolingRuntimeClientWrappper(Dispatcher dispatcher) {
    requireNonNull(dispatcher, "dispatcher cannot be null");

    this.dispatcher = dispatcher;
  }

  @Override
  public ToolingArtifact newToolingArtifact(URL artifactUrlContent, Map<String, String> properties) {
    Object toolingArtifactTarget = dispatcher.dispatchRemoteMethod("newToolingArtifact",
                                                                   ImmutableList.of(URL.class, Map.class),
                                                                   ImmutableList.of(serialize(artifactUrlContent),
                                                                                    serialize(properties)));
    return new ToolingArtifactWrapper(dispatcher.newReflectionInvoker(toolingArtifactTarget));
  }

  @Override
  public ToolingArtifact newToolingArtifact(URL artifactUrlContent, Map<String, String> properties, String parentId) {
    Object toolingArtifactTarget = dispatcher.dispatchRemoteMethod("newToolingArtifact",
                                                                   ImmutableList.of(URL.class, Map.class, String.class),
                                                                   ImmutableList.of(serialize(artifactUrlContent),
                                                                                    serialize(properties),
                                                                                    serialize(parentId)));
    return new ToolingArtifactWrapper(dispatcher.newReflectionInvoker(toolingArtifactTarget));
  }


  @Override
  public ToolingArtifact fetchToolingArtifact(String id) throws ToolingArtifactNotFoundException {
    Object toolingArtifactTarget = dispatcher.dispatchRemoteMethod("fetchToolingArtifact",
                                                                   ImmutableList.of(String.class),
                                                                   ImmutableList.of(serialize(id)));
    return new ToolingArtifactWrapper(dispatcher.newReflectionInvoker(toolingArtifactTarget));
  }

  @Override
  public ExtensionModelService extensionModelService() {
    Object extensionModelServiceTarget = dispatcher.dispatchRemoteMethod("extensionModelService");
    return new ExtensionModelServiceWrapper(dispatcher.newReflectionInvoker(extensionModelServiceTarget));
  }

  @Override
  public ArtifactSerializationService artifactSerializationService() {
    Object artifactSerializationServiceTarget = dispatcher.dispatchRemoteMethod("artifactSerializationService");
    return new ArtifactSerializationServiceWrapper(dispatcher.newReflectionInvoker(artifactSerializationServiceTarget));
  }

  @Override
  public DslSyntaxResolverService dslSyntaxResolverService() {
    Object dslSyntaxResolverServiceTarget = dispatcher.dispatchRemoteMethod("dslSyntaxResolverService");
    return new DslSyntaxResolverServiceWrapper(dispatcher.newReflectionInvoker(dslSyntaxResolverServiceTarget));
  }

  @Override
  public MessageHistoryService messageHistoryService() {
    Object messageHistoryServiceTarget = dispatcher.dispatchRemoteMethod("messageHistoryService");
    return new MessageHistoryServiceWrapper(dispatcher.newReflectionInvoker(messageHistoryServiceTarget));
  }

  @Override
  public ExtensionIconsService iconsService() {
    Object iconsServiceTarget = dispatcher.dispatchRemoteMethod("iconsService");
    return new IconsServiceWrapper(dispatcher.newReflectionInvoker(iconsServiceTarget));
  }

}
