/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.tooling.client.bootstrap.internal;

import static com.google.common.io.Files.createTempDir;
import static java.util.Optional.empty;
import static java.util.Optional.of;
import static org.hamcrest.core.Is.is;
import static org.junit.Assert.assertThat;
import static org.junit.rules.ExpectedException.none;
import static org.mockito.Matchers.any;
import static org.mockito.Mockito.mock;
import static org.mockito.Mockito.times;
import static org.mockito.Mockito.verify;
import static org.mockito.Mockito.when;
import static org.mule.maven.client.api.model.MavenConfiguration.newMavenConfigurationBuilder;
import static org.mule.tooling.client.bootstrap.api.ToolingRuntimeClientBootstrapFactory.newToolingVersionResolver;
import static org.mule.tooling.client.bootstrap.api.ToolingVersionResolverConfiguration.builder;
import static org.mule.tooling.client.bootstrap.api.UpdatePolicy.UPDATE_POLICY_INTERVAL;
import static org.mule.tooling.client.bootstrap.api.UpdatePolicy.UPDATE_POLICY_NEVER;
import org.mule.maven.client.api.MavenClient;
import org.mule.maven.client.api.VersionRangeResult;
import org.mule.maven.client.api.model.BundleDescriptor;
import org.mule.maven.client.api.model.RemoteRepository;
import org.mule.maven.client.internal.DefaultLocalRepositorySupplierFactory;
import org.mule.tooling.client.bootstrap.api.ToolingVersionResolver;
import org.mule.tooling.client.bootstrap.api.UpdatePolicy;

import java.io.File;
import java.io.IOException;
import java.net.URL;
import java.util.ArrayList;
import java.util.LinkedHashMap;
import java.util.List;

import org.apache.commons.io.FileUtils;
import org.junit.After;
import org.junit.Before;
import org.junit.Rule;
import org.junit.Test;
import org.junit.rules.ExpectedException;

public class DefaultToolingVersionResolverTestCase {

  private static final String MULESOFT_PUBLIC_REPOSITORY = "https://repository.mulesoft.org/nexus/content/repositories/public/";

  @Rule
  public ExpectedException expectedException = none();

  private File localRepository;

  @Before
  public void before() {
    localRepository = createTempDir();
  }

  @After
  public void after() {
    FileUtils.deleteQuietly(localRepository);
  }

  @Test
  public void nullWhenResolvingVersionNotAvailable() {
    File localMavenRepository = new DefaultLocalRepositorySupplierFactory().environmentMavenRepositorySupplier().get();


    ToolingVersionResolver versionResolver = newToolingVersionResolver(builder()
        .mavenConfiguration(
                            newMavenConfigurationBuilder()
                                .localMavenRepositoryLocation(localMavenRepository)
                                .build())
        .build());
    expectedException.expect(IllegalStateException.class);
    expectedException.expectMessage("There is no version mapped or available of Tooling Runtime Client for: '[0.0.0,0.1.0)'");
    versionResolver.resolve("0.0.1");
  }

  @Test
  public void resolveReleasedVersion() throws IOException {
    ToolingVersionResolver versionResolver = newToolingVersionResolver(builder()
        .mavenConfiguration(newMavenConfigurationBuilder()
            .localMavenRepositoryLocation(localRepository)
            .remoteRepository(RemoteRepository
                .newRemoteRepositoryBuilder()
                .id("releases")
                .url(new URL(MULESOFT_PUBLIC_REPOSITORY))
                .build())
            .build())
        .build());
    ToolingVersionResolver.ToolingVersion toolingVersion = versionResolver.resolve("4.0.0");
    assertThat(toolingVersion.getVersion(), is("4.0.0"));
    assertThat(toolingVersion.getPreviousVersion(), is(empty()));

    assertThat(versionResolver.getToolingVersionResolverConfiguration().dynamicMappings().get("[4.0.0,4.1.0)"), is("4.0.0"));
  }

  @Test
  public void resolveReleasedVersionAllowingSnapshots() throws IOException {
    ToolingVersionResolver versionResolver = newToolingVersionResolver(builder()
        .mavenConfiguration(newMavenConfigurationBuilder()
            .localMavenRepositoryLocation(localRepository)
            .remoteRepository(RemoteRepository
                .newRemoteRepositoryBuilder()
                .id("releases")
                .url(new URL(MULESOFT_PUBLIC_REPOSITORY))
                .build())
            .build())
        .allowedSuffixes("SNAPSHOT")
        .build());
    ToolingVersionResolver.ToolingVersion toolingVersion = versionResolver.resolve("4.0.0");
    assertThat(toolingVersion.getVersion(), is("4.1.0-SNAPSHOT"));
    assertThat(toolingVersion.getPreviousVersion(), is(empty()));
  }

  @Test
  public void resolveUsingMappings() throws IOException {
    File localMavenRepository = new DefaultLocalRepositorySupplierFactory().environmentMavenRepositorySupplier().get();

    LinkedHashMap<String, String> mappings = new LinkedHashMap<>();
    mappings.put("1.2.0", "4.0.102");
    mappings.put("1.3.2", "4.0.200");

    ToolingVersionResolver versionResolver = newToolingVersionResolver(builder()
        .mavenConfiguration(
                            newMavenConfigurationBuilder()
                                .localMavenRepositoryLocation(localMavenRepository)
                                .build())
        .mappings(mappings)
        .build());
    ToolingVersionResolver.ToolingVersion toolingVersion = versionResolver.resolve("1.3.2");
    assertThat(toolingVersion.getVersion(), is("4.0.200"));
    assertThat(toolingVersion.getPreviousVersion(), is(empty()));
  }

  @Test
  public void resolveUsingMappingsFixedVersionToOverrideRangeResolutionForSnapshots() throws IOException {
    File localMavenRepository = new DefaultLocalRepositorySupplierFactory().environmentMavenRepositorySupplier().get();

    LinkedHashMap<String, String> mappings = new LinkedHashMap<>();
    mappings.put("4.1.0-SNAPSHOT", "4.1.0-SNAPSHOT");
    mappings.put("[4.0.0,4.1.0)", "4.0.0");
    mappings.put("[4.1.0,4.2.0)", "4.1.0");

    ToolingVersionResolver versionResolver = newToolingVersionResolver(builder()
        .mavenConfiguration(
                            newMavenConfigurationBuilder()
                                .localMavenRepositoryLocation(localMavenRepository)
                                .build())
        .mappings(mappings)
        .build());
    ToolingVersionResolver.ToolingVersion toolingVersion = versionResolver.resolve("4.1.0-SNAPSHOT");
    assertThat(toolingVersion.getVersion(), is("4.1.0-SNAPSHOT"));
    assertThat(toolingVersion.getPreviousVersion(), is(empty()));
  }

  @Test
  public void resolveUsingPropertiesFileWithRanges() throws IOException {
    File localMavenRepository = new DefaultLocalRepositorySupplierFactory().environmentMavenRepositorySupplier().get();

    LinkedHashMap<String, String> mappings = new LinkedHashMap<>();
    mappings.put("1.2.0", "4.0.102");
    mappings.put("1.3.2", "4.0.200");
    mappings.put("(1.4.0,1.5.0]", "4.0.300");

    ToolingVersionResolver versionResolver = newToolingVersionResolver(builder()
        .mavenConfiguration(
                            newMavenConfigurationBuilder()
                                .localMavenRepositoryLocation(localMavenRepository)
                                .build())
        .mappings(mappings)
        .build());

    ToolingVersionResolver.ToolingVersion toolingVersion = versionResolver.resolve("1.4.3");
    assertThat(toolingVersion.getVersion(), is("4.0.300"));
    assertThat(toolingVersion.getPreviousVersion(), is(empty()));
  }

  @Test
  public void updatePolicyInterval() throws InterruptedException {
    MavenClient mavenClient = mock(MavenClient.class);
    VersionRangeResult versionRange = mock(VersionRangeResult.class);
    when(mavenClient.resolveVersionRange(any(BundleDescriptor.class))).thenReturn(versionRange);
    List<String> versions = new ArrayList<>();
    versions.add("4.0.0");
    when(versionRange.getVersions()).thenReturn(versions);

    ToolingVersionResolver versionResolver = new DefaultToolingVersionResolver(builder()
        .mavenConfiguration(newMavenConfigurationBuilder()
            .localMavenRepositoryLocation(localRepository)
            .build())
        // Set update policy to 1 minute
        .updatePolicy(new UpdatePolicy(UPDATE_POLICY_INTERVAL + ":1"))
        .build(), mavenClient);
    ToolingVersionResolver.ToolingVersion toolingVersion = versionResolver.resolve("4.0.0");
    assertThat(toolingVersion.getVersion(), is("4.0.0"));
    assertThat(toolingVersion.getPreviousVersion(), is(empty()));

    // Just sleep for more than 1 minute to trigger the update policy on next invocation
    Thread.sleep(61000);

    // Adding a new version (released)
    versions.add("4.0.1");

    toolingVersion = versionResolver.resolve("4.0.0");
    assertThat(toolingVersion.getVersion(), is("4.0.1"));
    assertThat(toolingVersion.getPreviousVersion(), is(of("4.0.0")));

    verify(mavenClient, times(2)).resolveVersionRange(any(BundleDescriptor.class));
  }

  @Test
  public void updatePolicyNever() throws InterruptedException {
    MavenClient mavenClient = mock(MavenClient.class);
    VersionRangeResult versionRange = mock(VersionRangeResult.class);
    when(mavenClient.resolveVersionRange(any(BundleDescriptor.class))).thenReturn(versionRange);
    List<String> versions = new ArrayList<>();
    versions.add("4.0.0");
    when(versionRange.getVersions()).thenReturn(versions);

    ToolingVersionResolver versionResolver = new DefaultToolingVersionResolver(builder()
        .mavenConfiguration(newMavenConfigurationBuilder()
            .localMavenRepositoryLocation(localRepository)
            .build())
        .updatePolicy(new UpdatePolicy(UPDATE_POLICY_NEVER))
        .build(), mavenClient);
    ToolingVersionResolver.ToolingVersion toolingVersion = versionResolver.resolve("4.0.0");
    assertThat(toolingVersion.getVersion(), is("4.0.0"));
    assertThat(toolingVersion.getPreviousVersion(), is(empty()));

    toolingVersion = versionResolver.resolve("4.0.0");
    assertThat(toolingVersion.getVersion(), is("4.0.0"));
    assertThat(toolingVersion.getPreviousVersion(), is(empty()));
    verify(mavenClient, times(1)).resolveVersionRange(any(BundleDescriptor.class));
  }

}
