/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
/*
* Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
* The software in this package is published under the terms of the CPAL v1.0
* license, a copy of which has been included with this distribution in the
* LICENSE.txt file.
*/

package org.mule.tooling.client.bootstrap.internal.wrapper;

import static java.util.Objects.requireNonNull;
import org.mule.maven.client.api.model.MavenConfiguration;
import org.mule.tooling.client.api.ToolingRuntimeClient;
import org.mule.tooling.client.api.configuration.agent.AgentConfiguration;
import org.mule.tooling.client.api.datasense.MetadataCacheFactory;
import org.mule.tooling.client.api.exception.ToolingException;
import org.mule.tooling.client.bootstrap.internal.reflection.Dispatcher;
import org.mule.tooling.client.internal.serialization.Serializer;

import com.google.common.collect.ImmutableList;

import java.lang.reflect.Method;

/**
 * Client side implementation for {@link ToolingRuntimeClient} that uses reflection and works as a bridge
 * between {@link ClassLoader classLoaders}.
 *
 * @since 1.0
 */
public class BuilderWrapper implements ToolingRuntimeClient.Builder {

  private final Dispatcher dispatcher;
  private ClassLoader toolingClassLoader;
  private Serializer serializer;

  public BuilderWrapper(ClassLoader toolingClassLoader, Dispatcher dispatcher, Serializer serializer) {
    requireNonNull(dispatcher, "dispatcher cannot be null");
    requireNonNull(toolingClassLoader, "toolingClassLoader cannot be null");
    requireNonNull(serializer, "serializer cannot be null");

    this.dispatcher = dispatcher;
    this.toolingClassLoader = toolingClassLoader;
    this.serializer = serializer;
  }

  @Override
  public ToolingRuntimeClient.Builder withRemoteAgentConfiguration(AgentConfiguration agentConfiguration) {
    dispatcher.dispatchRemoteMethod("withRemoteAgentConfiguration",
                                    ImmutableList.of(AgentConfiguration.class),
                                    ImmutableList.of(serializer.serialize(agentConfiguration)));
    return this;
  }

  @Override
  public ToolingRuntimeClient.Builder withMavenConfiguration(MavenConfiguration mavenConfiguration) {
    dispatcher.dispatchRemoteMethod("withMavenConfiguration",
                                    ImmutableList.of(MavenConfiguration.class),
                                    ImmutableList.of(serializer.serialize(mavenConfiguration)));
    return this;
  }

  @Override
  public ToolingRuntimeClient.Builder withMetadataCacheFactory(MetadataCacheFactory metadataCacheFactory) {
    // Different implementation as it has to create a proxy on implementation side that is able to invoke methods on the provided implementation.
    try {
      Class proxyMetadataCacheFactoryClazz =
          toolingClassLoader.loadClass("org.mule.tooling.client.internal.MetadataCacheFactoryProxy");
      String methodName = "withMetadataCacheFactory";
      Object proxyMetadataCacheFactory =
          proxyMetadataCacheFactoryClazz.getConstructor(Object.class).newInstance(metadataCacheFactory);
      Method method = dispatcher.findMethod(methodName);
      dispatcher.invoke(method, -1, new Object[] {proxyMetadataCacheFactory});
      return this;
    } catch (Exception e) {
      throw new ToolingException("Couldn't create proxy for MetadataCacheFactory", e);
    }
  }

  @Override
  public ToolingRuntimeClient build() {
    Object clientTarget = dispatcher.dispatchRemoteMethod("build");
    return new ToolingRuntimeClientWrappper(dispatcher.newReflectionInvoker(clientTarget), serializer);
  }

}
