/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */

package org.mule.tooling.client.bootstrap.internal.wrapper;

import static java.lang.String.format;
import static java.lang.Thread.currentThread;
import static java.util.Objects.requireNonNull;
import static java.util.ServiceLoader.load;
import org.mule.maven.client.api.model.MavenConfiguration;
import org.mule.tooling.client.api.Disposable;
import org.mule.tooling.client.api.ToolingRuntimeClient;
import org.mule.tooling.client.api.ToolingRuntimeClientBuilderFactory;
import org.mule.tooling.client.api.exception.ToolingException;
import org.mule.tooling.client.bootstrap.internal.reflection.Dispatcher;
import org.mule.tooling.client.internal.serialization.KryoClientSerializer;
import org.mule.tooling.client.internal.serialization.Serializer;
import org.mule.tooling.client.internal.serialization.XStreamClientSerializer;

import java.lang.reflect.InvocationTargetException;
import java.util.ServiceLoader;
import java.util.concurrent.ExecutorService;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Client side implementation for {@link ToolingRuntimeClientBuilderFactory} that uses reflection and works as a bridge
 * between {@link ClassLoader classLoaders}.
 *
 * @since 1.0
 */
public class ToolingRuntimeClientBuilderFactoryWrapper implements ToolingRuntimeClientBuilderFactory, Disposable {

  private static Logger LOGGER = LoggerFactory.getLogger(ToolingRuntimeClientBuilderFactoryWrapper.class);

  private String muleVersion;
  private ClassLoader toolingClassLoader;
  private Dispatcher dispatcher;
  private Serializer serializer;
  private MavenConfiguration mavenConfiguration;

  public ToolingRuntimeClientBuilderFactoryWrapper(String muleVersion, ClassLoader toolingClassLoader,
                                                   ExecutorService executorService, MavenConfiguration mavenConfiguration) {
    requireNonNull(muleVersion, "muleVersion cannot be null");
    requireNonNull(toolingClassLoader, "toolingClassLoader cannot be null");
    requireNonNull(executorService, "executorService cannot be null");
    requireNonNull(mavenConfiguration, "mavenConfiguration cannot be null");

    this.muleVersion = muleVersion;
    this.toolingClassLoader = toolingClassLoader;

    Object toolingRuntimeClientFactory = discoverToolingRuntimeClientFactory();
    this.serializer = discoverSerialization(toolingRuntimeClientFactory);
    this.dispatcher = new Dispatcher(toolingRuntimeClientFactory, toolingClassLoader, executorService);
    this.mavenConfiguration = mavenConfiguration;
  }

  private Serializer discoverSerialization(Object toolingRuntimeClientFactory) {
    try {
      toolingRuntimeClientFactory.getClass().getMethod("setSerialization", String.class).invoke(toolingRuntimeClientFactory,
                                                                                                KryoClientSerializer.NAME);
      LOGGER.info("Using Kryo serialization");
      return new KryoClientSerializer();
    } catch (NoSuchMethodException e) {
      // Kryo is not supported by the bootstrapped instance
      LOGGER.info("Using xStream serialization, as API supports Kryo but not the bootstrapped implementation");
      return new XStreamClientSerializer();
    } catch (IllegalAccessException e) {
      throw new ToolingException("Error while discovering serialization", e);
    } catch (InvocationTargetException e) {
      throw new ToolingException("Error while discovering serialization", e);
    }
  }

  private Object discoverToolingRuntimeClientFactory() {
    final ClassLoader contextClassLoader = currentThread().getContextClassLoader();
    try {
      currentThread().setContextClassLoader(toolingClassLoader);
      // Tooling Runtime Client 4.1.x supports validation of minMuleVersion (correctly) based on changes on Bootstrapping
      return toolingClassLoader.loadClass("org.mule.tooling.client.internal.DefaultToolingRuntimeClientFactory")
          .getConstructor(new Class[] {String.class}).newInstance(muleVersion);
    } catch (NoSuchMethodException e) {
      // Tooling Runtime Client 4.0.x would keep on working with SPI and use Mule Version from Mule Core jar manifest
      try {
        ServiceLoader<? extends Object> serviceLoader =
            load(toolingClassLoader.loadClass(ToolingRuntimeClientBuilderFactory.class.getName()), toolingClassLoader);
        if (!serviceLoader.iterator().hasNext()) {
          throw new IllegalStateException("No service found for: '" + ToolingRuntimeClientBuilderFactory.class.getName()
              + "'");
        }

        return serviceLoader.iterator().next();
      } catch (ClassNotFoundException e1) {
        throw new IllegalStateException(format(
                                               "ClassLoader for Tooling was not correctly created as there is no implementation of %s",
                                               ToolingRuntimeClientBuilderFactory.class.getName()),
                                        e1);
      }
    } catch (Exception e) {
      throw new IllegalStateException(format(
                                             "Error while creating the implementation of %s",
                                             ToolingRuntimeClientBuilderFactory.class.getName()),
                                      e);
    } finally {
      currentThread().setContextClassLoader(contextClassLoader);
    }
  }

  @Override
  public void dispose() {
    dispatcher.dispatchRemoteMethod("dispose");
  }

  @Override
  public ToolingRuntimeClient.Builder create() {
    Object builderTarget = dispatcher.dispatchRemoteMethod("create");
    ToolingRuntimeClient.Builder builder =
        new BuilderWrapper(toolingClassLoader, dispatcher.newReflectionInvoker(builderTarget), serializer);
    // By default it mavenConfiguration from bootstrap is inherited
    builder.withMavenConfiguration(mavenConfiguration);
    return builder;
  }

}
