/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */

package org.mule.tooling.client.bootstrap.api;

import java.util.Optional;

/**
 * Resolver to allow clients to determinate the Tooling Runtime Client version for a given Mule Runtime version.
 * <p/>
 *
 * This resolver takes into account an {@link ToolingVersionResolverConfiguration#mappings()} to resolve versions of
 * Tooling Runtime Client for the requested Mule Runtime version.
 * The mappings should have as key an specific Mule Runtime version or a Maven range version and value an specific version Tooling Runtime Client.
 * <p/>
 *
 * In case an entry is not defined on {@link ToolingVersionResolverConfiguration#mappings()} the resolver will do a to look up for
 * the latest version of Tooling Runtime Client for the {@code major.minor} version of the Mule Runtime version passed as parameter.
 * Once a version for Tooling Runtime Client is resolved using Maven, it is cached until the
 * {@link ToolingVersionResolverConfiguration#remoteVersionUpdatePolicy()} is reached, at that point will check if a new version is
 * available from Maven remote repositories which in that case the old one will be returned as part of the response.
 * <p/>
 *
 * For example:
 * <ul>
 *  <li>Day 0: Mule Runtime 4.0.0, Tooling Runtime Client 4.0.0<br>
 *      {@code resolve("4.0.0") -> 4.0.0}
 *  </li>
 *  <li>Day 7: Mule Runtime 4.0.0,4.0.1,4.0.2, Tooling Runtime Client 4.0.0,4.0.1<br>
 *      {@code resolve("4.0.0") -> 4.0.1}<br>
 *      {@code resolve("4.0.1") -> 4.0.1}<br>
 *      {@code resolve("4.0.2") -> 4.0.1}
 *  </li>
 *  <li>Day 14: Mule Runtime 4.0.0,4.0.1,4.0.2, Tooling Runtime Client 4.0.0,4.0.1,4.0.2,4.0.3<br>
 *        {@code resolve("4.0.0") -> 4.0.3}<br>
 *        {@code resolve("4.0.1") -> 4.0.3}<br>
 *        {@code resolve("4.0.2") -> 4.0.3}
 *  </li>
 *  <li>Day 21: Mule Runtime 4.0.0,4.0.1,4.0.2, Tooling Runtime Client 4.0.0,4.0.1,4.0.2,4.0.3<br>
 *        But setting a mapping for [4.0.0,4.1.0)=4.0.2<br>
 *        {@code resolve("4.0.0") -> 4.0.2}<br>
 *        {@code resolve("4.0.1") -> 4.0.2}<br>
 *        {@code resolve("4.0.2") -> 4.0.2}
 *  </li>
 * </ul>
 *
 * @see <a href="https://cwiki.apache.org/confluence/display/MAVENOLD/Dependency+Mediation+and+Conflict+Resolution#DependencyMediationandConflictResolution-DependencyVersionRanges">
 * https://cwiki.apache.org/confluence/display/MAVENOLD/Dependency+Mediation+and+Conflict+Resolution#DependencyMediationandConflictResolution-DependencyVersionRanges</a>
 * 
 * @since 1.1
 */
public interface ToolingVersionResolver {

  /**
   * @param muleVersion a given Mule Runtime version.
   * @return the Tooling Runtime Client version that supports the Mule Runtime version or null if there is no one mapped or not resolved when using a Maven range.
   */
  ToolingVersion resolve(String muleVersion);

  /**
   * @return the {@link DynamicToolingVersionResolverConfiguration} in order to let clients know the status of the dynamic mappings resolved
   * using Maven.
   */
  DynamicToolingVersionResolverConfiguration getToolingVersionResolverConfiguration();

  /**
   * The result of resolving a Tooling Runtime Client version.
   *
   * @since 1.1
   */
  interface ToolingVersion {

    /**
     * @return version resolved for the requested Mule Runtime version.
     */
    String getVersion();

    /**
     * @return empty if the version resolved came out from the mappings or the version was resolved for the first time using Maven.
     * If the version was already resolved using Maven and the update policy reached, if the new resolved version is different the old
     * one cached will be returned here. This will allow clients to know that for a given Mule Runtime there was an updated Tooling
     * Runtime Client version available and they could dispose the previous one.
     */
    Optional<String> getPreviousVersion();

  }
}
