/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */

package org.mule.tooling.client.bootstrap.api;

import org.mule.maven.client.api.model.MavenConfiguration;
import org.mule.tooling.client.bootstrap.internal.DefaultToolingVersionResolverConfigurationBuilder;

import java.util.LinkedHashMap;
import java.util.List;
import java.util.Properties;

/**
 * Defines the configuration for {@link ToolingVersionResolver}.
 * <p/>
 * The {@link #mappings()} if defined will be the first place to check for a Tooling Runtime Client version whenever the
 * {@link ToolingVersionResolver#resolve(String)} is called. If there is not a matching Mule Runtime version Maven
 * would be used to get the latest Tooling Runtime Client version for the given Mule Runtime.
 * <p/>
 * By design Tooling Runtime Client will match the same {@code major.minor} version of Mule Runtime.
 *
 * @since 1.1
 */
public interface ToolingVersionResolverConfiguration {

  /**
   * @return {@link MavenConfiguration} to retrieve the versions for a range of Tooling Runtime Client.
   */
  MavenConfiguration mavenConfiguration();

  /**
   * @return {@link UpdatePolicy} for updating remote versions resolved for Tooling Runtime Client using Maven. Default is {@link UpdatePolicy#UPDATE_POLICY_DAILY}.
   */
  UpdatePolicy remoteVersionUpdatePolicy();

  /**
   * @return when resolving versions depending on what was deployed on the configured Maven repositories the list of versions
   * could contain also versions with suffix, for instance, {@code 4.1.0-SNAPSHOT} or {@code 4.0.0-beta-1}. On these cases those versions
   * would be included in the list of versions resolved for {@code [4.0.0,4.1.0)} therefore this allows to define a whitelist for the suffix
   * that should be considered.
   */
  List<String> allowedSuffixes();

  /**
   * @return {@link LinkedHashMap<String, String>} with the set of mappings versions. Keys are Mule Runtime version or ranges in Maven format and value
   * is a Tooling Runtime Client version. The mapping entries will be considered in order, useful when needed to do a mapping for
   * an SNAPSHOT version and at the same time there is already a range defined in mapping for that version.
   * This takes precedence over the Maven remote resolution.
   */
  LinkedHashMap<String, String> mappings();

  static ToolingVersionResolverConfigurationBuilder builder() {
    return new DefaultToolingVersionResolverConfigurationBuilder();
  }

  /**
   * Builder for {@link ToolingVersionResolverConfiguration}.
   *
   * @since 1.0
   */
  interface ToolingVersionResolverConfigurationBuilder {

    /**
     * @param mavenConfiguration {@link MavenConfiguration} to resolve the latest Tooling Runtime Client.
     */
    ToolingVersionResolverConfigurationBuilder mavenConfiguration(MavenConfiguration mavenConfiguration);

    /**
     * @param updatePolicy {@link UpdatePolicy} for updating remote versions resolved for Tooling Runtime Client using Maven.
     */
    ToolingVersionResolverConfigurationBuilder updatePolicy(UpdatePolicy updatePolicy);

    /**
     * @param allowedSuffixes {@link List} of suffixes that are allowed when resolving latest version of Tooling Runtime Client.
     */
    ToolingVersionResolverConfigurationBuilder allowedSuffixes(String... allowedSuffixes);

    /**
     * The resolver will take into account this mappings to resolve versions of Tooling Runtime Client for the requested Mule Runtime version first.
     * The mappings should have as key an specific Mule Runtime version or a Maven range version and value an specific version Tooling Runtime Client.
     * <p/>
     * If the mappings don't contain a mapped version of Tooling Runtime Client for the requested Mule Runtime version the resolver will
     * use Maven to fetch the latest version for the given Mule Runtime.
     * @param mappings {@link Properties} with the set of mappings versions. Keys are Mule Runtime version or ranges in Maven format and value
     * is a Tooling Runtime Client version.
     */
    ToolingVersionResolverConfigurationBuilder mappings(LinkedHashMap<String, String> mappings);

    /**
     * @return builds the {@link ToolingVersionResolverConfiguration}.
     */
    ToolingVersionResolverConfiguration build();

  }

}
