/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */

package org.mule.tooling.client.bootstrap.internal;

import static java.lang.String.format;
import static java.util.Optional.ofNullable;
import static org.mule.tooling.client.internal.util.Preconditions.checkNotNull;
import static org.mule.tooling.client.internal.util.Preconditions.checkState;
import org.mule.maven.client.api.model.MavenConfiguration;
import org.mule.tooling.client.bootstrap.api.ExecutorServiceConfiguration;
import org.mule.tooling.client.bootstrap.api.ToolingRuntimeClientBootstrapConfiguration;

import com.vdurmont.semver4j.Semver;
import com.vdurmont.semver4j.SemverException;

import java.io.File;
import java.net.URI;
import java.util.Optional;

/**
 * Default implementation of {@link ToolingRuntimeClientBootstrapConfiguration.ToolingRuntimeClientBootstrapConfigurationBuilder}.
 */
public class DefaultToolingRuntimeClientBootstrapConfigurationBuilder implements
    ToolingRuntimeClientBootstrapConfiguration.ToolingRuntimeClientBootstrapConfigurationBuilder {

  private String muleVersion;
  private String toolingVersion;
  private MavenConfiguration mavenConfiguration;
  private URI log4jConfiguration;
  private ExecutorServiceConfiguration executorServiceConfiguration;
  private File workingFolder;

  @Override
  public ToolingRuntimeClientBootstrapConfiguration.ToolingRuntimeClientBootstrapConfigurationBuilder muleVersion(
                                                                                                                  String muleVersion) {
    this.muleVersion = muleVersion;
    return this;
  }

  @Override
  public ToolingRuntimeClientBootstrapConfiguration.ToolingRuntimeClientBootstrapConfigurationBuilder toolingVersion(
                                                                                                                     String toolingVersion) {
    this.toolingVersion = toolingVersion;
    return this;
  }

  @Override
  public ToolingRuntimeClientBootstrapConfiguration.ToolingRuntimeClientBootstrapConfigurationBuilder mavenConfiguration(
                                                                                                                         MavenConfiguration mavenConfiguration) {
    this.mavenConfiguration = mavenConfiguration;
    return this;
  }

  @Override
  public ToolingRuntimeClientBootstrapConfiguration.ToolingRuntimeClientBootstrapConfigurationBuilder log4jConfiguration(
                                                                                                                         URI log4jConfiguration) {
    this.log4jConfiguration = log4jConfiguration;
    return this;
  }

  @Override
  public ToolingRuntimeClientBootstrapConfiguration.ToolingRuntimeClientBootstrapConfigurationBuilder executorServiceConfiguration(
                                                                                                                                   ExecutorServiceConfiguration executorServiceConfiguration) {
    this.executorServiceConfiguration = executorServiceConfiguration;
    return this;
  }

  @Override
  public ToolingRuntimeClientBootstrapConfiguration.ToolingRuntimeClientBootstrapConfigurationBuilder workingFolder(
                                                                                                                    File workingFolder) {
    this.workingFolder = workingFolder;
    return this;
  }

  @Override
  public ToolingRuntimeClientBootstrapConfiguration build() {
    return new DefaultToolingRuntimeClientBootstrapConfiguration(muleVersion, toolingVersion, mavenConfiguration,
                                                                 log4jConfiguration,
                                                                 executorServiceConfiguration,
                                                                 workingFolder);
  }

  private static class DefaultToolingRuntimeClientBootstrapConfiguration implements ToolingRuntimeClientBootstrapConfiguration {

    private String muleVersion;
    private String toolingVersion;
    private MavenConfiguration mavenConfiguration;
    private URI log4jConfiguration;
    private ExecutorServiceConfiguration executorServiceConfiguration;
    private File workingFolder;

    DefaultToolingRuntimeClientBootstrapConfiguration(String muleVersion, String toolingVersion,
                                                      MavenConfiguration mavenConfiguration,
                                                      URI log4jConfiguration,
                                                      ExecutorServiceConfiguration executorServiceConfiguration,
                                                      File workingFolder) {
      checkNotNull(muleVersion, "muleVersion cannot be null");
      checkNotNull(toolingVersion, "toolingVersion cannot be null");
      checkNotNull(mavenConfiguration, "mavenConfiguration cannot be null");

      checkState(validateSemanticVersioning(muleVersion),
                 format("muleVersion '%s' has to be defined with 'major.minor.patch' format", muleVersion));
      checkState(validateSemanticVersioning(toolingVersion),
                 format("toolingVersion '%s' has to be defined with 'major.minor.patch' format", toolingVersion));

      this.muleVersion = muleVersion;
      this.toolingVersion = toolingVersion;
      this.mavenConfiguration = mavenConfiguration;
      this.log4jConfiguration = log4jConfiguration;
      this.executorServiceConfiguration = executorServiceConfiguration;
      this.workingFolder = workingFolder;
    }

    private boolean validateSemanticVersioning(String version) {
      try {
        new Semver(version, Semver.SemverType.STRICT);
        return true;
      } catch (SemverException e) {
        return false;
      }
    }

    @Override
    public String muleVersion() {
      return muleVersion;
    }

    @Override
    public String toolingVersion() {
      return toolingVersion;
    }

    @Override
    public MavenConfiguration mavenConfiguration() {
      return mavenConfiguration;
    }

    @Override
    public Optional<URI> log4jConfiguration() {
      return ofNullable(log4jConfiguration);
    }

    @Override
    public Optional<ExecutorServiceConfiguration> executorServiceConfiguration() {
      return ofNullable(executorServiceConfiguration);
    }

    @Override
    public File workingFolder() {
      return workingFolder;
    }
  }
}
